/*************************************************************************************************************
 * Universal USB to I2S Interface Indicator
 *
 * Project Page: http://www.dimdim.gr/arduino/universal-usb-to-i2s-interface-indicator/
 *
 * v1.37    08/11/2016 : - Changed once again DSD detection code for Amanero. Now properly supports DSD256/512.
 *
 * v1.35    13/11/2015 : - Changed DSD detection code for Amanero. Now supports DSD64/128/256/512.
 * 
 * v1.34    26/06/2015 : - Added support for SH1106-based OLED displays.
 *
 * v1.33    26/06/2015 : - Added code to send any non-recognized IR codes to the serial port. 
 *
 * v1.28    07/06/2015 : - JLsounds code is fully tested and working OK.
 *
 * v1.23    01/06/2015 : - Remote power on/off implemented.
 *                       - JLsounds code is not working! To be fixed soon..
 *
 * v1.03    29/05/2015 : - Signal type is now displayed (for 3 seconds).
 *                       - IR remote control supported for changing sources.
 *                       - Source name is displayed (in case of USB for 3 seconds).
 *
 * v0.36    25/05/2015 : - Updated WaveIO code to also display 352.8K & 384K. It is now fully tested.
 *                       - Fixed decimal point display bug.
 *                       - Amanero code is fully tested and working OK.
 *
 * v0.34    26/04/2015 : - Sampling Rate now displayed as float (with decimal point). Slightly smaller font.
 *
 * v0.33    24/04/2015 : - Debugged Amanero code. Should be fine now. Still not tested.
 *
 * v0.32    22/04/2015 : - Added DSD detection for the JLSounds interface (not properly documented, 
 *                         so not sure if it works OK).
 *                       - Added support for WaveIO interface (not tested).
 *
 * v0.3     21/04/2015 : - Initial version. Shows only the sampling rate (type INT for now).
 *                       - Compatible with 0.96" OLED display (SSD1306, 128 x 64)
 *                       - Compatible with Amanero Combo384 (not tested), DIYINHK XMOS & 
 *                         JLSounds interfaces (not tested).
 *
 **************************************************************************************************************/


#include "U8glib.h"
#include <stdlib.h>
#include <IRremote.h>

// Remove comment slashes ("//") from your USB interface. Keep all other interfaces commented out.
#define Amanero
// #define DIYINHK_XMOS
// #define JLSounds
// #define WaveIO

// Set the operation of the U2 isolator. If you have a 4-input part (Si8045AA), comment out the TWOOUTPUTS. 
// If you have a 2-input/2-output part (Si8642ΒΑ), comment out the ALLINPUTS.
//#define ALLINPUTS
#define TWOOUTPUTS

// Comment out when not using the IR control functionality.
//#define IRCONTROL

// setup u8g object - select according to your specific OLED
//U8GLIB_SSD1306_128X64 u8g(U8G_I2C_OPT_NO_ACK);	                // 0.96" OLED Display using I2C which does not send ACK
//U8GLIB_SH1106_128X64 u8g(U8G_I2C_OPT_NONE);                   // I2C / TWI 
//U8GLIB_SH1106_128X64 u8g(U8G_I2C_OPT_DEV_0|U8G_I2C_OPT_FAST); // Dev 0, Fast I2C / TWI
U8GLIB_SH1106_128X64 u8g(U8G_I2C_OPT_NO_ACK);                 // Display which does not send ACK

#define POWER_CODE 0xFF48B7          // Code for power on/off
#define SOURCE1_CODE 0xFF827D        // Code for source 1
#define SOURCE2_CODE 0xFFB24D        // Code for source 2

int x_dim;                           // Display resolution
int y_dim;                           // Display resolution

int str_height;
int str_width;
int source = 1;                      // Variable to hold the Input number. 0 for s/pdif, 1 for USB.
int sourceold = 3;                   // Variable to hold the old Input number. 0 for s/pdif, 1 for USB.
int type;                            // Variable to hold the signal type. 0 for PCM, 1 for DSD.
int typeold = 3;                     // Variable to hold the old signal type. 0 for PCM, 1 for DSD.
bool poweron = true;                 // Variable to hold the poweron condition.
bool newSR;                          // Tell if the SR has changed.
bool newSource;                      // Tell if the source has changed.
bool display_SR;
bool display_type;
bool display_power;
bool display_source;
float SR;                            // Variable to hold the detected sampling rate (or DSD type) as float.
float SRold = 1.1;                   // Variable to hold the old detected sampling rate (or DSD type) as float.
int SRint;                           // Variable to hold the detected sampling rate (or DSD type) as int.
int SRintold;                        // Variable to hold the old detected sampling rate (or DSD type) as int.
char stringSR[10];                   // Variable to hold the detected sampling rate (or DSD type) as string.
unsigned long previousMillis = 0;
long power_disp_interval = 3000;
long source_disp_interval = 3000;

const int I1 = 2;          // I1 is Digital 2 Pin.
const int I2 = 3;          // I2 is Digital 3 Pin.
const int I3 = 4;          // I3 is Digital 4 Pin.
const int I4 = 5;          // I4 is Digital 5 Pin.
const int I5 = 6;          // I5 is Digital 6 Pin.
const int I6 = 7;          // I6 is Digital 7 Pin.
const int I7 = 8;          // I7 is Digital 8 Pin.
const int I8 = 9;          // I8 is Digital 9 Pin.
const int I9 = 10;         // I9 is Digital 10 Pin.
const int I10 = 11;        // I10 is Digital 11 Pin.

int RECV_PIN = 12;         // IR Receiver input pin.

#define POWERPIN 13        // Power Relay pin.

#ifdef IRCONTROL
IRrecv irrecv(RECV_PIN);
decode_results results;
#endif IRCONTROL

void disp_SR(void) {
  
  u8g.setFont(u8g_font_fur35n);
  if ((SR-(int)SR) > 0)
  {
    str_height = u8g.getFontAscent()-u8g.getFontDescent();
    str_width = u8g.getStrWidth(dtostrf(SR, 0, 1, stringSR));
    
    u8g.setPrintPos((x_dim-str_width)/2, (y_dim/2+str_height/2));
    u8g.print(SR, 1);
  }
  else if ((SR-(int)SR) == 0)
  {
    SRint = (int)SR;
    str_height = u8g.getFontAscent()-u8g.getFontDescent();
    str_width = u8g.getStrWidth(itoa(SRint, stringSR, 10));
    
    u8g.setPrintPos((x_dim-str_width)/2, (y_dim/2+str_height/2));
    u8g.print(SRint, 1);
  }
}

void disp_source(void)
  {
    u8g.setFont(u8g_font_fub30);
    str_height = u8g.getFontAscent()-u8g.getFontDescent();

    if (source == 0)
    {
      str_width = u8g.getStrWidth("S/PDIF");
      u8g.setPrintPos((x_dim-str_width)/2, (y_dim/2+str_height/2));
      u8g.print("S/PDIF");
      //Serial.println("S/PDIF");
    }
    else
    {
      str_width = u8g.getStrWidth("USB");
      u8g.setPrintPos((x_dim-str_width)/2, (y_dim/2+str_height/2));
      u8g.print("USB");
      //Serial.println("USB");
    }
   
  }

void disp_type(void)
  {
    u8g.setFont(u8g_font_fub30);
    str_height = u8g.getFontAscent()-u8g.getFontDescent();
   
    if (type == 0)
    {
      str_width = u8g.getStrWidth("PCM");
      u8g.setPrintPos((x_dim-str_width)/2, (y_dim/2+str_height/2));
      u8g.print("PCM");
    }
    else
    {
      str_width = u8g.getStrWidth("DSD");
      u8g.setPrintPos((x_dim-str_width)/2, (y_dim/2+str_height/2));
      u8g.print("DSD");
    }
  }
  

void disp_power(void)
  {
    u8g.setFont(u8g_font_fub30);
    str_height = u8g.getFontAscent()-u8g.getFontDescent();
   
    if (poweron == false)
    {
      str_width = u8g.getStrWidth("OFF");
      u8g.setPrintPos((x_dim-str_width)/2, (y_dim/2+str_height/2));
      u8g.print(" ");
    }
    else
    {
      str_width = u8g.getStrWidth("ON");
      u8g.setPrintPos((x_dim-str_width)/2, (y_dim/2+str_height/2));
      u8g.print("ON");
    }
  }
  
void read_input() {

  // ---------------------------- Amanero ------------------------------------------------------------------------------
  #ifdef Amanero
  /*
   * I1 = F0
   * I2 = F1
   * I3 = F2
   * I4 = F3
   * I5 = DSD_ON
   * I6 = DSD 64 or 128
   */
  // Check for PCM or DSD
  if (digitalRead(I5) == false)    // PCM detected
  {  
  type = 0;
  if (digitalRead(I1) == false && digitalRead(I2) == false && digitalRead(I3) == false && digitalRead(I4) == false)
    {
      SR = 32;
    }
    else if (digitalRead(I1) == true && digitalRead(I2) == false && digitalRead(I3) == false && digitalRead(I4) == false)
      {
        SR = 44.1;
      }
      else if (digitalRead(I1) == false && digitalRead(I2) == true && digitalRead(I3) == false && digitalRead(I4) == false)
        {
          SR = 48;
        }
        else if (digitalRead(I1) == true && digitalRead(I2) == true && digitalRead(I3) == false && digitalRead(I4) == false)
          {
            SR = 88.2;
          }
          else if (digitalRead(I1) == false && digitalRead(I2) == false && digitalRead(I3) == true && digitalRead(I4) == false)
            {
              SR = 96;
            }
            else if (digitalRead(I1) == true && digitalRead(I2) == false && digitalRead(I3) == true && digitalRead(I4) == false)
              {
                SR = 176.4;
              }
              else if (digitalRead(I1) == false && digitalRead(I2) == true && digitalRead(I3) == true && digitalRead(I4) == false)
                {
                  SR = 192;
                }
                else if (digitalRead(I1) == true && digitalRead(I2) == true && digitalRead(I3) == true && digitalRead(I4) == false)
                  {
                    SR = 352.8;
                  }
                  else if (digitalRead(I1) == false && digitalRead(I2) == false && digitalRead(I3) == false && digitalRead(I4) == true)
                    {
                      SR = 384;
                    }
  }
  else if (digitalRead(I5) == true)    // DSD detected
    {
      type = 1;
   if (digitalRead(I4) == true && digitalRead(I3) == false && digitalRead(I2) == false && digitalRead (I1) == true)
        {
          SR = 64;
        }
        else if (digitalRead(I4) == true && digitalRead(I3) == false && digitalRead(I2) == true && digitalRead (I1) == false)
          {
            SR = 128;
          }
          else if (digitalRead(I4) == true && digitalRead(I3) == false && digitalRead(I2) == true && digitalRead (I1) == true)
            {
              SR = 256;
            }
              else if (digitalRead(I4) == true && digitalRead(I3) == true && digitalRead(I2) == false && digitalRead (I1) == false)
              {
                SR = 512;
              }     
    }
  #endif Amanero

  // ---------------------------- DIYINHK_XMOS ------------------------------------------------------------------------------
  #ifdef DIYINHK_XMOS
  if (digitalRead(I1) == false && digitalRead(I2) == false && digitalRead(I3) == false && digitalRead(I4) == false)
      {
        type = 0;
        SR = 44.1;
      }
      else if (digitalRead(I1) == true && digitalRead(I2) == false && digitalRead(I3) == false && digitalRead(I4) == false)
        {
          type = 0;
          SR = 48;
        }
        else if (digitalRead(I1) == false && digitalRead(I2) == true && digitalRead(I3) == false && digitalRead(I4) == false)
          {
            type = 0;
            SR = 88.2;
          }
          else if (digitalRead(I1) == true && digitalRead(I2) == true && digitalRead(I3) == false && digitalRead(I4) == false)
            {
              type = 0;
              SR = 96;
            }
            else if (digitalRead(I1) == false && digitalRead(I2) == false && digitalRead(I3) == true && digitalRead(I4) == false)
              {
                type = 0;
                SR = 176.4;
              }
              else if (digitalRead(I1) == true && digitalRead(I2) == false && digitalRead(I3) == true && digitalRead(I4) == false)
                {
                  type = 0;
                  SR = 192;
                }
                else if (digitalRead(I1) == false && digitalRead(I2) == true && digitalRead(I3) == true && digitalRead(I4) == false)
                  {
                    type = 0;
                    SR = 352.8;
                  }
                  else if (digitalRead(I1) == true && digitalRead(I2) == true && digitalRead(I3) == true && digitalRead(I4) == false)
                    {
                      type = 0;
                      SR = 384;
                    }
                    else if (digitalRead(I1) == false && digitalRead(I2) == false && digitalRead(I3) == true && digitalRead(I4) == true)
                      {
                        type = 1;
                        SR = 64;
                      }
                      else if (digitalRead(I1) == true && digitalRead(I2) == false && digitalRead(I3) == true && digitalRead(I4) == true)
                        {
                          type = 1;
                          SR = 128;
                        }
  #endif DIYINHK_XMOS
  
  // ---------------------------- JLsounds ------------------------------------------------------------------------------
  #ifdef JLSounds
  // Check for PCM or DSD
  if (digitalRead(I4) == false)    // PCM detected
  {  
    type = 0;
    if (digitalRead(I1) == false && digitalRead(I2) == false && digitalRead(I3) == false)
      {
        SR = 44.1;
      }
      else if (digitalRead(I1) == true && digitalRead(I2) == false && digitalRead(I3) == false)
        {
          SR = 48;
        }
        else if (digitalRead(I1) == false && digitalRead(I2) == true && digitalRead(I3) == false)
          {
            SR = 88.2;
          }
          else if (digitalRead(I1) == true && digitalRead(I2) == true && digitalRead(I3) == false)
            {
              SR = 96;
            }
            else if (digitalRead(I1) == false && digitalRead(I2) == false && digitalRead(I3) == true)
              {
                SR = 176.4;
              }
              else if (digitalRead(I1) == true && digitalRead(I2) == false && digitalRead(I3) == true)
                {
                  SR = 192;
                }
                else if (digitalRead(I1) == false && digitalRead(I2) == true && digitalRead(I3) == true)
                  {
                    SR = 352.8;
                  }
                  else if (digitalRead(I1) == true && digitalRead(I2) == true && digitalRead(I3) == true)
                    {
                      SR = 384;
                    }
  }
  else if (digitalRead(I4) == true)    // DSD detected
    {
      type = 1;
      if (digitalRead(I1) == false && digitalRead(I2) == false && digitalRead(I3) == true)
        {
          SR = 64;
        }
          else if (digitalRead(I1) == false && digitalRead(I2) == true && digitalRead(I3) == true)
            {
              SR = 128;
            }
    }
 
  #endif JLSounds

  // ---------------------------- WaveIO ------------------------------------------------------------------------------
  #ifdef WaveIO
  type = 0;
  if (digitalRead(I1) == true && digitalRead(I2) == false && digitalRead(I3) == false && digitalRead(I4) == false && digitalRead(I5) == false && digitalRead(I6) == false)
      {
        SR = 44.1;
      }
      else if (digitalRead(I1) == false && digitalRead(I2) == true && digitalRead(I3) == false && digitalRead(I4) == false && digitalRead(I5) == false && digitalRead(I6) == false)
        {
          SR = 48;
        }
        else if (digitalRead(I1) == false && digitalRead(I2) == false && digitalRead(I3) == true && digitalRead(I4) == false && digitalRead(I5) == false && digitalRead(I6) == false)
          {
            SR = 88.2;
          }
          else if (digitalRead(I1) == false && digitalRead(I2) == false && digitalRead(I3) == false && digitalRead(I4) == true && digitalRead(I5) == false && digitalRead(I6) == false)
            {
              SR = 96;
            }
            else if (digitalRead(I1) == false && digitalRead(I2) == false && digitalRead(I3) == false && digitalRead(I4) == false && digitalRead(I5) == true && digitalRead(I6) == false)
              {
                SR = 176.4;
              }
              else if (digitalRead(I1) == false && digitalRead(I2) == false && digitalRead(I3) == false && digitalRead(I4) == false && digitalRead(I5) == false && digitalRead(I6) == true)
                {
                  SR = 192;
                }
                else if (digitalRead(I1) == false && digitalRead(I2) == false && digitalRead(I3) == true && digitalRead(I4) == false && digitalRead(I5) == true && digitalRead(I6) == false)
                  {
                    SR = 352.8;
                  }
                  else if (digitalRead(I1) == false && digitalRead(I2) == false && digitalRead(I3) == false && digitalRead(I4) == true && digitalRead(I5) == false && digitalRead(I6) == true)
                    {
                      SR = 384;
                    }
  #endif WaveIO
}

void setup(void) {

  pinMode(I1, INPUT);
  digitalWrite(I1, LOW);
  pinMode(I2, INPUT);
  digitalWrite(I2, LOW);
  pinMode(I3, INPUT);
  digitalWrite(I3, LOW);
  pinMode(I4, INPUT);
  digitalWrite(I4, LOW);
  pinMode(I5, INPUT);
  digitalWrite(I5, LOW);
  pinMode(I6, INPUT);
  digitalWrite(I6, LOW);
  
  #ifdef ALLINPUTS
  pinMode(I7, INPUT);
  digitalWrite(I7, LOW);
  pinMode(I8, INPUT);
  digitalWrite(I8, LOW);
  pinMode(I9, INPUT);
  digitalWrite(I9, LOW);
  pinMode(I10, INPUT);
  digitalWrite(I10, LOW);
  #endif ALLINPUTS
  
  #ifdef TWOOUTPUTS
  pinMode(I7, INPUT);
  digitalWrite(I7, LOW);
  pinMode(I8, INPUT);
  digitalWrite(I8, LOW);
  pinMode(I9, OUTPUT);
  digitalWrite(I9, LOW);
  pinMode(I10, OUTPUT);
  digitalWrite(I10, LOW);
  #endif TWOOUTPUTS
  
  pinMode(POWERPIN, OUTPUT);
  digitalWrite(POWERPIN, LOW);
  
  x_dim = u8g.getWidth();
  y_dim = u8g.getHeight();
    
  Serial.begin(9600);
  
  #ifdef IRCONTROL
  irrecv.enableIRIn();                       // Start the IR receiver
  //Serial.println("IR on");
  poweron = false;
  #endif IRCONTROL
}

void loop()
{
  #ifdef IRCONTROL
  if (irrecv.decode(&results)) {            // If IR code is received, get the IR code.
    if (results.value == POWER_CODE) {      // What to do if the Power button is pressed
        //Serial.write("Power\n");
        if (poweron == false)                 // Check if already powered on
          {
            digitalWrite(POWERPIN, HIGH);
            poweron = true;
            display_power = true;
            unsigned long currentMillis = millis();
            if ((currentMillis - previousMillis) > source_disp_interval)
              {
                previousMillis = currentMillis;

                u8g.firstPage();                          // picture loop
                do {
                    disp_source();
                    } while(u8g.nextPage());
                display_SR = false;
                display_type = false;
                //Serial.println("displaying source");
                delay(2000);
              }
            else 
              {
                sourceold = source;
                display_SR = false;
              }  
          }
          else 
          {
            digitalWrite(POWERPIN, LOW);
            poweron = false;
            
            u8g.firstPage();                          // picture loop
              do {
                disp_power();
              } while(u8g.nextPage());
          }
    } 
    else if ((results.value == SOURCE1_CODE) && (poweron == true)) {      // What to do if the Source 1 button is pressed
        //Serial.write("Input 1\n");
        digitalWrite(I9, LOW);
        digitalWrite(I10, LOW);
        source = 0;
    }
    
    else if ((results.value == SOURCE2_CODE) && (poweron == true)) {      // What to do if the Source 2 button is pressed
        //Serial.write("Input 2\n");
        digitalWrite(I9, HIGH);
        digitalWrite(I10, HIGH);
        source = 1;
    }
       
    else {
      Serial.print("unexpected value: ");
      Serial.println(results.value, HEX);
   }
  irrecv.resume();                                 // Resume decoding (necessary!)
  }
  #endif IRCONTROL
  
  read_input();                              // Scan the USB Interface Module for changes in SR and/or format
     
  unsigned long currentMillis = millis();

      if ((source != sourceold) && ((currentMillis - previousMillis) > source_disp_interval) && (poweron == true))
        {
          previousMillis = currentMillis;

          u8g.firstPage();                          // picture loop
            do {
              disp_source();
            } while(u8g.nextPage());
          display_SR = false;
          display_type = false;
          //Serial.println("displaying source");
          delay(2000);
        }
      else 
      {
        sourceold = source;
        if ((source == 1) && (poweron == true))
        {
          display_SR = true;
        }
     }

      if ((type != typeold) && ((currentMillis - previousMillis) > source_disp_interval) && (poweron == true) && (source == 1))
        {
          previousMillis = currentMillis;

          u8g.firstPage();                          // picture loop
          do {
               disp_type();
             } while(u8g.nextPage());
             
          display_SR = false;
          //Serial.println("displaying type");
          delay(2000);
        }
      else 
      {
        typeold = type;
        if ((source == 1) && (poweron == true))
        {
          display_SR = true;
        }
     }
  
  if ((display_SR == true) && (poweron == true))
    {
      u8g.firstPage();                          // picture loop
      do {
        disp_SR();
      } while(u8g.nextPage());
    }
}

