/********************************************************************************************************
 * TFT HiFiDUINO Pro Code: Controlling the ES9038Pro and ES9028Pro with an Arduino DUE with a colour TFT.
 *
 * Project Page: http://www.dimdim.gr/tft-hifiduino-pro-project/
 *
 * v1.09    17/10/2017 : - Fixed small backlight bug
 * 
 * v1.08    15/10/2017 : - Fixed EEPROM init bug
 *                       - Fixed DPLL settings bug & default DPLL settings for USB 2
 *                       - Added alternative way of controlling Solid State Sidecar (via Pin A1)
 *                       - Changed input names & icons to match my Buffalo III
 *                       - Other minor bug fixes
 *                       - Inverted power-on signal to make fully compatible with "TFT HiFiDuino" code
 *                       - Inverted rotation of encoder to make fully compatible with "TFT HiFiDuino" code
 * 
 * v1.06    23/07/2017 : - First public release
 *
 ********************************************************************************************************/

#include "Wire.h"                               // Library for the I2C communications
#include "Adafruit_MCP23008.h"                  // Library for the I2C port expander
#include <UTFT.h>                               // Library for the TFT
#include <UTFT_DLB.h>                           // Extension for the UTFT library
#include <IRremote2.h>                          // Library for the IR remote
#include <RotaryEncoder.h>                      // Library for the encoders
#include "extEEPROM.h"                          // Library for the I2C EEPROM

// I2C EEPROM stuff
extEEPROM myEEPROM(kbits_256, 1, 64, 0x50);     // myEEPROM object, 256kbit, 1 device on the bus, 64 bytes page size, 0x50 

// TFT stuff
// Declare which fonts we will be using
extern uint8_t Shruti_Bold_num_48x70[];
extern uint8_t calibri_bold_80[];
extern uint8_t calibri_bold_60[];
extern uint8_t calibri_bold_40[];
extern uint8_t calibri_bold_26[];
extern uint8_t calibri_bold_20[];

// Set whether you will be using a classic (3.2") or a BIG (4.3") TFT. Comment out if using 3.2" TFT.
//#define BIG

#ifdef BIG
  UTFT_DLB myGLCD(ITDB43,38,39,40,41);
  int tft_x = 480;
  int tft_y = 272;
#else
  UTFT_DLB myGLCD(ITDB32WD,38,39,40,41);
  int tft_x = 400;
  int tft_y = 240;
#endif BIG

// Set whether you will be using a power on/off relay. Comment out ALWAYSON if you will be using a remote for power on/off.
//#define ALWAYSON

#ifdef ALWAYSON
  bool poweron = true;        // Variable to hold whether the DAC is in power-on state or not
#else
  bool poweron = false;       // Variable to hold whether the DAC is in power-on state or not
#endif ALWAYSON

uint16_t xoffset,yoffset;

//----------------------------- Bitmaps ----------------------------------------/
extern unsigned short usb[0x2710];
extern unsigned short toslink[0x2710];
extern unsigned short rca[0x2710];
extern unsigned short xlr[0x2710];
extern unsigned short dsd[0x2710];
extern unsigned short pcm[0x2710];
extern unsigned short spdif[0x2710];
extern unsigned short nolock[0x2710];

extern unsigned short usb_b[0x2710];
extern unsigned short toslink_b[0x2710];
extern unsigned short rca_b[0x2710];
extern unsigned short xlr_b[0x2710];
extern unsigned short dsd_b[0x2710];
extern unsigned short pcm_b[0x2710];
extern unsigned short spdif_b[0x2710];
extern unsigned short nolock_b[0x2710];

// Remote control codes. They correspond to an old remote that I use for testing - change to match your remote's.
#define POWER_CODE 0xFF48B7      // Code for power on/off
#define VOLUP_CODE 0xFF30CF      // Code for Volume up
#define VOLDOWN_CODE 0xFF609F    // Code for Volume down
#define MUTE_CODE  0xFF40BF      // Code for mute
#define SETTINGS_CODE 0xFFD02F   // Code for DAC Settings ("Setup" button on my remote)
#define GLOBAL_SET_CODE 0xFFD827 // Code for global DAC Settings ("Title" button on my remote)
#define SELECT_CODE 0xFF02FD     // Code for Select button
#define LEFT_CODE 0xFF9867       // Code for left arrow
#define RIGHT_CODE 0xFF8877      // Code for right arrow
#define SOURCE1_CODE 0xFF827D    // Code for source 1
#define SOURCE2_CODE 0xFFB24D    // Code for source 2
#define SOURCE3_CODE 0xFFA25D    // Code for source 3
#define SOURCE4_CODE 0xFF42BD    // Code for source 4
#define SOURCE5_CODE 0xFF728D    // Code for source 5
#define SOURCE6_CODE 0xFF629D    // Code for source 6
#define SOURCE7_CODE 0xFFC23D    // Code for source 7
#define SOURCE8_CODE 0xFFF20D    // Code for source 8
#define SOURCE9_CODE 0xFFE21D    // Code for source 9
#define SOURCE10_CODE 0xFF00FF   // Code for source 10

// Set whether you want black characters on white background (default) or white characters on black background (#define WHITE).
#define WHITE

byte dac = 0x48;                     // address of DAC chip
int dac_chip = 0;                    // Type of DAC chip for auto detection. 0 = ES9038Pro, 1 = ES9028Pro

#define MCLK 10                      // Value of Clock used (in 10s of MHz). 10 = 100MHz.

byte RECV_PIN = 9;                   // IR Receiver input pin.
IRrecv irrecv(RECV_PIN);
decode_results results;
int prev_result;

volatile int dir;                    // Rotary encoder variable.
volatile int rotPosition;            // Rotary encoder variable.
volatile int oldRotPosition;         // Rotary encoder variable.

int VOLUPPIN=6;                      // RotEnc A terminal for right rotary encoder.
int VOLDOWNPIN=7;                    // RotEnc B terminal for right rotary encoder.
#define SELECTPIN 5                  // Rotary Encoder button pin.

int VOLUPPIN2=A4;                    // RotEnc A terminal for left rotary encoder.
int VOLDOWNPIN2=A3;                  // RotEnc B terminal for left rotary encoder.
#define SELECTPIN2 A5                // Switch to select function for left rotary encoder.

int INTERRUPTPIN=3;                  // Connect to Interrupt pin on DAC (GPIO4).
int SIDECARPIN=A1;                   // Connect to Solid State Sidecar control pin (not to be used with the original TPA Sidecar!!! Danger of damage!!).
int POWERPIN=A0;                     // Power relay pin.
int DIMMPIN=8;                       // Pin for TFT backlight controll. A LOW on this pin turns on the backlight.

Adafruit_MCP23008 mcp;               // Set up the I/O expander.

// Button stuff
unsigned long keyPrevMillis = 0;
const unsigned long keySampleIntervalMs = 25;
byte longKeyPressCountMax = 80;            // Duration of button press to be considered "long". 80 * 25 = 2000 ms
byte mediumKeyPressCountMin = 40;          // Duration of button press to be considered "medium". 40 * 25 = 1000 ms
byte KeyPressCount = 0;
byte prevKeyState = HIGH;                  // button is active low
const byte keyPin = SELECTPIN;             // button is connected to pin SELECTPIN and GND
int key = 0;                               // Variable to hold the state of the button

boolean select1Mode=false;                 // To indicate whether in Input options select mode or not
boolean selectDPLLMode=false;              // To indicate whether in DPLL settings select mode or not
boolean select2Mode=false;                 // To indicate whether in settings select mode or not
unsigned long select1Millis = 0;           // Stores last recorded time for being in select mode
unsigned long selectDPLLMillis = 0;        // Stores last recorded time for being in DPLL select mode
unsigned long select2Millis = 0;           // Stores last recorded time for being in select mode

byte select1 = 1;                          // To record first level select position (FIL, VOL, DPLL, etc)
int selectD = 0;                           // To record DPLL select position
byte select2 = 1;                          // To record second level (setup) select position

// The order of selection when clicking the select switch
#define FIR 1           // FIR filter selection, also number of param variable
#define IIRV 2          // IIR filter selection, also number of param variable
#define LCK 3           // Lock speed setting, also number of param variable
#define DTH 4           // Dither setting, also number of param variable
#define JTR 5           // Jitter eliminator selection, also number of param variable
#define OSF 6           // OSF setting, also number of param variable
#define DMF 7           // Deemphasis enabled or disabled, also number of param variable
#define DPL 8           // DPLL setting menu, also number of param variable

#define MAXPARAM 10     // Total number of parameters to keep track of for each input.
#define MAXPARAM2 4     // Total number of parameters to keep track of for Settings menu.
#define TOTALINPUTS 5   // Total number of inputs (from 0 to...).

int param = 0;          // Hold the number of the parameter to be changed
int paramDPLL = 0;      // Hold the number of the DPLL parameter to be changed
int MAXPARAMDPLL = 17;  // Total number of DPLL parameters to keep track of.
int setting = 0;        // Hold the number of the settings parameter to be changed

// The order of selection when clicking the select switch while in Settings mode
#define PRE 1           // Preamp mode
#define PRE_VOL 2       // Default volume when in Preamp mode
#define BLGHT 3         // TFT Backlight level

#define INTERVAL_SELECT 6                  // Time in sec to exit select mode when no activity

RotaryEncoder encoder(VOLDOWNPIN, VOLUPPIN);       // Setup the first Rotary Encoder
RotaryEncoder encoder2(VOLDOWNPIN2, VOLUPPIN2);    // Setup the second Rotary Encoder. Not used at the moment.
static int pos = 0;

int irdir=0;
int eeAddress=0;                           // EEPROM address
int eeAddress1=0;                          // EEPROM address of Input 1 memory
int eeAddress2=128;                        // EEPROM address of Input 2 memory
int eeAddress3=256;                        // EEPROM address of Input 3 memory
int eeAddress4=384;                        // EEPROM address of Input 4 memory
int eeAddress5=512;                        // EEPROM address of Input 5 memory
int eeAddress6=640;                        // EEPROM address of Input 6 memory
int eeAddressS=1920;                       // EEPROM address of Settings memory
int eeAddressT=2048;                       // EEPROM address of test data

// Variables for DAC chip parameters (specific to certain input).
int volume = 0;
int volumeOld = 0;
int input = 0;
int inputOld = 0;
int filter = 0;
int filterOld = 0;
int IIR = 0;
int mute = 0;
int DPLL_PCM = 5;
int DPLL_DSD = 5;
bool Deemphasis_en = 0;
bool Deemphasis_auto = 0;
int Deemphasis_sel = 0;
int LockSpeed = 0;
bool Dither = 1;
bool JitterElim = 1;
bool bypass_osf = 0;
bool deemph_bypass = 0;
bool auto_deemph = 0;
int deemph_sel = 0;
int SR = 0;
int SR_Old = 0;
int sig_type = 0;
int sig_typeOld = 0;
bool lock = 0;
bool lockOld = 0;

// Variables for Settings parameters.
int preamp = 1;
int preamp_def_vol = 50;
int backlight = 10;

bool do_select_inp = 0;
bool do_select_filter = 0;
bool select_inp_blue = 0;

// Variables that store the width of each parameter , to aid with "erasing" the text.
int param_1_width;
int param_2_width;
int param_3_width;
int param_4_width;
int param_5_width;
int param_6_width;
int param_7_width;
int param_8_width;

// Variables that store the state of selection of each parameter.
bool param_1_select;
bool param_2_select;
bool param_3_select;
bool param_4_select;
bool param_5_select;
bool param_6_select;
bool param_7_select;
bool param_8_select;

// Variables that store the width of each setting, to aid with "erasing" the text.
int setting_1_width;
int setting_2_width;
int setting_3_width;

// Variables that store the state of selection of each setting.
bool setting_1_select;
bool setting_2_select;
bool setting_3_select;

// Variables that store the width of each DPLL parameter, to aid with "erasing" the text.
int paramDPLL_16_width;
int paramDPLL_15_width;
int paramDPLL_14_width;
int paramDPLL_13_width;
int paramDPLL_12_width;
int paramDPLL_11_width;
int paramDPLL_10_width;
int paramDPLL_9_width;
int paramDPLL_8_width;
int paramDPLL_7_width;
int paramDPLL_6_width;
int paramDPLL_5_width;
int paramDPLL_4_width;
int paramDPLL_3_width;
int paramDPLL_2_width;
int paramDPLL_1_width;

// Variables that store the state of selection of each DPLL parameter.
bool paramDPLL_16_select;
bool paramDPLL_15_select;
bool paramDPLL_14_select;
bool paramDPLL_13_select;
bool paramDPLL_12_select;
bool paramDPLL_11_select;
bool paramDPLL_10_select;
bool paramDPLL_9_select;
bool paramDPLL_8_select;
bool paramDPLL_7_select;
bool paramDPLL_6_select;
bool paramDPLL_5_select;
bool paramDPLL_4_select;
bool paramDPLL_3_select;
bool paramDPLL_2_select;
bool paramDPLL_1_select;

byte error;

// Character array to store the names of the 7 filters.
#ifdef BIG
  char* myFilters[]={"Fast Roll-off, Linear Phase", "Slow Roll-off, Linear Phase", "Fast Roll-off, Minimum Phase", 
  "Slow Roll-off, Minimum Phase", "Apodizing, Fast, Linear Phase", "Hybrid, Fast, Minimum Phase", "Brickwall Filter"};
#else
  char* myFilters[]={"Fast Roll-off, Linear Ph.", "Slow Roll-off, Linear Ph.", "Fast Roll-off, Minimum Ph.", 
  "Slow Roll-off, Minimum Ph.", "Apodizing, Fast, Linear Ph.", "Hybrid, Fast, Minimum Ph.", "Brickwall Filter"};
#endif BIG

// Character array to store the names of the inputs.
char* myInputs[]={"USB 1", "USB 2", "COAX", "TOSLINK 1", "TOSLINK 2", "AES/EBU", 0};

// Character array to store the names of the signal types.
char* mySignal[]={"DoP", "SPDIF", "PCM", "DSD", 0};

// Character array to store the names of the DSD types or Sampling Rates.
// SR = 0 = Invalid, SR = 1..5 = DSD, SR = 6..15 = PCM
char* mySR[]={"Invalid", "DSD1024", "DSD512", "DSD256", "DSD128", "DSD64", "768KHz", "705.6KHz", "384KHz", "352.8KHz", 
              "192KHz", "176.4KHz", "96KHz", "88.2KHz", "48KHz", "44.1KHz", "32KHz", 0};

// Character array to store the names of the DPLL settings for PCM signals.
// DPLLPCM = 0 = DPLL Off, DPLLPCM = 1 = Lowest, etc.
char* myDPLLPCM[]={"DPLL Off", "Lowest", "Low", "Default", "Med-Low", "Medium", "High", "Higher", "Highest", 0};

// Character array to store the names of the DPLL settings for DSD signals.
// DPLLDSD = 0 = DPLL Off, DPLLDSD = 1 = Lowest, etc.
char* myDPLLDSD[]={"DPLL Off", "Lowest", "Low", "Med-Low", "Medium", "Med-High", "Default", "High", "Higher", "Highest", 0};

// Structure that holds the main parameters for each of the inputs.
struct DAC_Input 
  {
    int filter_val;         // Filter number.
    int IIR_val;            // IIR filter number.
    int DPLLPCM1;           // DPLL value for 768KHz.
    int DPLLPCM2;           // DPLL value for 705.6KHz.
    int DPLLPCM3;           // DPLL value for 384KHz.
    int DPLLPCM4;           // DPLL value for 352.8KHz or DoP 128.
    int DPLLPCM5;           // DPLL value for 192KHz.
    int DPLLPCM6;           // DPLL value for 176.4KHz or DoP 64.
    int DPLLPCM7;           // DPLL value for 96KHz.
    int DPLLPCM8;           // DPLL value for 88.2KHz.
    int DPLLPCM9;           // DPLL value for 48KHz.
    int DPLLPCM10;          // DPLL value for 44.1KHz.
    int DPLLPCM11;          // DPLL value for 32KHz.
    int DPLLDSD1;           // DPLL value for native DSD1024.
    int DPLLDSD2;           // DPLL value for native DSD512.
    int DPLLDSD3;           // DPLL value for native DSD256.
    int DPLLDSD4;           // DPLL value for native DSD128.
    int DPLLDSD5;           // DPLL value for native DSD64.
    int lockspeed_val;      // Lock speed setting.
    bool dither_val;        // Dither enabled or disabled.
    bool jitter_elim_val;   // Jitter Eliminator enabled or disabled.
    bool bypass_osf_val;    // OSF enabled or disabled.
    bool deemph_bypass_val; // De-emphasis enabled or disabled.
    bool auto_deemph_val;   // Auto de-emphasis enabled or disabled.
    int deemph_sel_val;     // De-emphasis filter selection.
  };

DAC_Input Input[TOTALINPUTS];              // Declare structures of Inputs

// Structure that holds the DAC's general settings.
struct DAC_Settings
  {
    bool variable;          // Variable or fixed output.
    int preamp_def_vol;     // Default power-on volume when set for variable output.
    int backlight;          // TFT backlight value.
  };

DAC_Settings Settings1;     // Declare structure of Settings1

// Structure that holds initialization test content.
struct InitTest
  {
    int test1;
    int test2;
    int test3;
  };

InitTest EpromTest;         // Declare structure of initialization data

// ----------------------------------------------- Beginning of setup()---------------------------------------------------------------
void setup() 
  {
 
  SerialUSB.begin(115200);           // for debugging

  // Start communication with the I2C EEPROM chip
  byte i2cStat = myEEPROM.begin(myEEPROM.twiClock100kHz);
  if ( i2cStat != 0 )
    {
      SerialUSB.println(F("Problem communicating with EEPROM chip."));
    }
    
  // Set up MCP23008 pins
  mcp.begin();                    // use default address 0
  mcp.pinMode(0, OUTPUT);
  mcp.pinMode(1, OUTPUT);
  mcp.pinMode(2, OUTPUT);
  mcp.pinMode(3, OUTPUT);
  mcp.pinMode(4, OUTPUT);
  mcp.pinMode(5, OUTPUT);
  mcp.pinMode(6, OUTPUT);
  mcp.pinMode(7, OUTPUT);

  // Set up the TFT
  myGLCD.InitLCD();
  myGLCD.clrScr();
  myGLCD.setBackColor(0, 0, 0);

  // Set up the pin modes
  pinMode(VOLUPPIN, INPUT);       // Button switch or Encoder pin for volume up
  digitalWrite(VOLUPPIN, HIGH);   // If H/W debouncing is implemented, set to LOW

  pinMode(VOLDOWNPIN, INPUT);     // Button switch or Encoder pin for volume down
  digitalWrite(VOLDOWNPIN, HIGH); // If H/W debouncing is implemented, set to LOW

  pinMode(SELECTPIN, INPUT_PULLUP);      // Button switch or Encoder pin for Select
  
  pinMode(POWERPIN, OUTPUT);      // Power relay control pin
  digitalWrite(POWERPIN, HIGH);    // Keep high on powerup

  pinMode(INTERRUPTPIN, INPUT);   // Interrupt pin

  pinMode(SIDECARPIN, OUTPUT);     // SSS control pin
  digitalWrite(SIDECARPIN, LOW);   // By default select I2S input

  
  irrecv.enableIRIn();            // Start the IR receiver
  SerialUSB.println(F("IR on"));

  Wire.begin();                   // Join The I2C Bus As A Master 

  if (poweron == true)
    {
      DeviceInit();                   // Initialize the DAC chip
      setInput();                     // Set the input to the default input
      //main_disp(true);
    }

  }

// ----------------------------------------------- Beginning of main loop()---------------------------------------------------------------
void loop() 
{
// ----------------------------------------------- Check lock status by monitoring interrupt pin -----------------------------------------

  if (digitalRead(INTERRUPTPIN)==HIGH)
    {
      getLockStatus();
      getSR();
      getSR();
      if ((select1Mode==false) && (select2Mode==false) && (selectDPLLMode==false))
        {
          main_disp(false);
        }
    }

// ----------------------------------------------- Rotary Encoder Code -------------------------------------------------------------------
  
  int dir = 0;
  encoder.tick();
  int newPos = encoder.getPosition();
  if (pos != newPos) 
    {
      if (pos < newPos)
        {
          dir = 2;
          pos = newPos;
        }
      else if (pos > newPos)
        {
          dir = 1;
          pos = newPos;
        }
    }

  
  if((dir==1 || dir==2 || irdir==1 || irdir==2) && select1Mode==false && select2Mode==false && selectDPLLMode==false)     // When not in select1Mode
  {
      if (dir==1 || irdir==1)  // CW
      {
        //SerialUSB.println("");
        //SerialUSB.print("do_select_inp: "); SerialUSB.println(do_select_inp);
        //SerialUSB.print("do_select_filter: "); SerialUSB.println(do_select_filter);
        //SerialUSB.print("Preamp: "); SerialUSB.println(preamp);
        //SerialUSB.println("");
        //SerialUSB.println("CW");
        if (((do_select_inp == 1) && (do_select_filter == 0)) || ((preamp == false) && (do_select_filter == 0)))        // If in input select state or just not in preamp mode, switch inputs
          {
            if (input<TOTALINPUTS)
            {
              input++;
            }
            else input = 0;
            SerialUSB.print(F("Input: ")); SerialUSB.println(input);
            setInput();
            main_disp(false);
            irdir=0;
            select1Millis=millis();
          }
        else if ((do_select_filter == 1) && (preamp == false))
          {
            if (filter < 6)
              {
                filter++;
              }
            else filter = 0;
            setFilter();
            main_disp(false);
            savetoEEPROM();
            irdir=0;
            select1Millis=millis();
          }
        else
          {
            if ((volume > 0) && (preamp == true))             // If in preamp mode, change volume
              {
                volume=volume-2;
                setVol();
                main_disp(false);
                irdir=0;                
              }
          }
      }
      else                                                    // If not CW, then it is CCW
      {
        //SerialUSB.println("CCW");
        if (((do_select_inp == 1) && (do_select_filter == 0)) || ((preamp == false) && (do_select_filter == 0)))        // If in input select state or just not in preamp mode, switch inputs
          {
            if (input>0)
            {
              input--;
            }
            else input = TOTALINPUTS;
            SerialUSB.print(F("Input: ")); SerialUSB.println(input);            
            setInput();
            main_disp(false);
            irdir=0;
            select1Millis=millis();            
          }
        else if ((do_select_filter == 1) && (preamp == false))
          {
            if (filter > 0)
              {
                filter--;
              }
            else filter = 6;
            setFilter();
            main_disp(false);
            savetoEEPROM();
            irdir=0;
            select1Millis=millis();
          }
        else
          {
            if ((volume < 254) && (preamp == true))             // If in preamp mode, change volume
              {
                volume=volume+2;
                setVol();
                main_disp(false);
                irdir=0;
              }
          }
      }
    dir=0;
  }  // End of not-in-select1Mode

  if((dir==1 || dir==2 || irdir==1 || irdir==2) && select1Mode==true)     // When in select1Mode
  {
    switch(select1%(MAXPARAM))
      {  
      case FIR:
      if (dir==1 || irdir==1)  // CW
        {
          if (filter < 6)
            {
              filter++;
            }
          else filter = 0;
        }
      else                                                      // If not CW, then it is CCW
        {
          if (filter > 0)
            {
              filter--;
            }
          else filter = 6;
        }
      param=FIR;
      setFilter();
      menu1_disp(false);
      savetoEEPROM();
      select1Millis=millis();
      irdir=0;
      break;
      
      case IIRV:
      if (dir==1 || irdir==1)  // CW
        {
          if (IIR < 3)
            {
              IIR++;
            }
          else IIR = 0;
        }
      else                     // If not CW, then it is CCW
        {
          if (IIR > 0)
            {
              IIR--;
            }
          else IIR = 3;
        }
      param=IIRV;
      setIIR();
      menu1_disp(false);
      savetoEEPROM();
      select1Millis=millis();
      irdir=0;
      break;

      case LCK:
      if (dir==1 || irdir==1)  // CW
        {
          if (LockSpeed > 0)
            {
              LockSpeed--;
            }
          else LockSpeed = 4;
        }
      else                     // If not CW, then it is CCW
        {
          if (LockSpeed < 4)
            {
              LockSpeed++;
            }
          else LockSpeed = 0;
        }
      param=LCK;
      setLockSpeed();
      menu1_disp(false);
      savetoEEPROM();
      select1Millis=millis();
      irdir=0;
      break;

      case DTH:
      if (dir==1 || irdir==1)  // CW
        {
          if (Dither == true)
            {
              Dither = false;
            }
          else if (Dither == false)
            {
              Dither = true;
            }
        }
      else                     // If not CW, then it is CCW
        {
          if (Dither == false)
            {
              Dither = true;
            }
          else if (Dither == true)
            {
              Dither = false;
            }
        }
      param=DTH;
      setDither();
      menu1_disp(false);
      savetoEEPROM();
      select1Millis=millis();
      irdir=0;
      break;

      case JTR:
      if (dir==1 || irdir==1)  // CW
        {
          if (JitterElim == true)
            {
              JitterElim = false;
            }
          else if (JitterElim == false)
            {
              JitterElim = true;
            }
        }
      else                     // If not CW, then it is CCW
        {
          if (JitterElim == false)
            {
              JitterElim = true;
            }
          else if (JitterElim == true)
            {
              JitterElim = false;
            }
        }
      setJitterElim();
      param=JTR;
      menu1_disp(false);
      savetoEEPROM();
      select1Millis=millis();
      irdir=0;
      break;
      
      case OSF:
      if (dir==1 || irdir==1)  // CW
        {
          if (bypass_osf == true)
            {
              bypass_osf = false;
            }
          else if (bypass_osf == false)
            {
              bypass_osf = true;
            }
        }
      else                     // If not CW, then it is CCW
        {
          if (bypass_osf == false)
            {
              bypass_osf = true;
            }
          else if (bypass_osf == true)
            {
              bypass_osf = false;
            }            
        }
      setOSF();
      savetoEEPROM();
      param=OSF;
      menu1_disp(false);
      select1Millis=millis();
      irdir=0;
      break;

      case DMF:
      if (dir==1 || irdir==1)                                     // CW
        {
          if (deemph_bypass == true)                              // if de-emphasis is disabled
            {
              deemph_bypass = false;                              // Enable de-emphasis
              auto_deemph = true;                                 // Set to auto
            }
          else if ((deemph_bypass == false) && (auto_deemph == true))
            {
              auto_deemph = false;
              deemph_sel = 0;
            }
          else if ((deemph_bypass == false) && (auto_deemph == false) && (deemph_sel == 0))
            {
              deemph_sel = 1;
            }            
          else if ((deemph_bypass == false) && (auto_deemph == false) && (deemph_sel == 1))
            {
              deemph_sel = 2;
            }
          else if ((deemph_bypass == false) && (auto_deemph == false) && (deemph_sel == 2))
            {
              deemph_bypass = true;
            }            
        }
      else                                                        // If not CW, then it is CCW
        {
          if (deemph_bypass == true)
            {
              deemph_bypass = false;
              auto_deemph = false;
              deemph_sel = 2;
            }
          else if ((deemph_bypass == false) && (auto_deemph == false) && (deemph_sel == 2))
            {
              deemph_sel = 1;
            }
          else if ((deemph_bypass == false) && (auto_deemph == false) && (deemph_sel == 1))
            {
              deemph_sel = 0;
            }
          else if ((deemph_bypass == false) && (auto_deemph == false) && (deemph_sel == 0))
            {
              auto_deemph = true;
              deemph_sel = 0;
            }         
          else if ((deemph_bypass == false) && (auto_deemph == true))
            {
              deemph_bypass = true;
            }                 
        }
      param=DMF;
      setDeemphasis();
      menu1_disp(false);
      savetoEEPROM();
      select1Millis=millis();
      irdir=0;      
      break;

      case DPL:
      if (dir==1 || dir==2 || irdir==1 || irdir==0)                                     // CW 
        {
          param=DPL;
          if ((input == 0) || (input == 1))
            {
              paramDPLL=1;
              selectD=1;
            }
          else 
            {
              paramDPLL=5;
              selectD=5;
            }
          select1Mode=false;
          selectDPLLMode=true;
          selectDPLLMillis=millis();
          SerialUSB.println(F("Go into DPLL Settings menu"));
          menu_dpll_disp(true);
          irdir=0;      
          break;
        }
    }
    dir=0;
  }  // End of "in select1Mode"

  if((dir==1 || dir==2 || irdir==1 || irdir==2) && select2Mode==true)     // When in select2Mode
  {
    switch(select2%(MAXPARAM2))
      {  
      case PRE:
      if (dir==1 || irdir==1)  // CW
        {
          if (preamp == 1)
            {
              preamp = 0;
              volume = 0;
              setVol();
            }
          else if (preamp == 0)
            {
              preamp = 1;
              volume = Settings1.preamp_def_vol;
              setVol();              
            }
        }
      else                                                              // If not CW, then it is CCW
        {
          if (preamp == 1)
            {
              preamp = 0;
              volume = 0;
              setVol();
            }
          else if (preamp == 0)
            {
              preamp = 1;
              volume = Settings1.preamp_def_vol;
              setVol();              
            }
        }
      setting=PRE;
      settings_disp(false);
      savetoEEPROM();
      select2Millis=millis();
      irdir=0;
      break;
      
      case PRE_VOL:
      if (dir==1 || irdir==1)  // CW
        {
          if (preamp_def_vol > 0)
            {
              preamp_def_vol=preamp_def_vol-2;
              if (preamp == true)
                {
                  volume = preamp_def_vol;
                  setVol();               
                }
            }
        }
      else                                                              // If not CW, then it is CCW
        {
          if (preamp_def_vol < 254)
            {
              preamp_def_vol=preamp_def_vol+2;
              if (preamp == true)
                {
                  volume = preamp_def_vol;
                  setVol();               
                }              
            }
        }
      setting=PRE_VOL;
      settings_disp(false);
      savetoEEPROM();
      select2Millis=millis();
      irdir=0;
      break;

      case BLGHT:
      if (dir==1 || irdir==1)                                         // CW
        {
          //SerialUSB.print("Backlight: "); SerialUSB.println(backlight);
          if (backlight < 10)
            {
              backlight++;
            }
        }
      else                                                            // If not CW, then it is CCW
        {
          if (backlight > 1)
            {
              backlight--;
            }
        }
      setting=BLGHT;
      setBacklight();
      settings_disp(false);
      savetoEEPROM();
      select2Millis=millis();
      irdir=0;
      break;
    }
    dir=0;
  }                                                                 // End of "in select2Mode"

  if((dir==1 || dir==2 || irdir==1 || irdir==2) && selectDPLLMode==true)     // When in selectDPLLMode
  {
    //SerialUSB.print(F("SelectD (rotary): ")); SerialUSB.println(selectD);
    if (selectD == 0)
      {
        selectD = 16;
      }
    switch(selectD)
      {  
      case 1:                                                   // PCM 768K
      if (dir==1 || irdir==1)  // CW
        {
          if (Input[input].DPLLPCM1 < 8)
            {
              Input[input].DPLLPCM1++;
            }
          else Input[input].DPLLPCM1 = 0;
        }
      else                                                      // If not CW, then it is CCW
        {
          if (Input[input].DPLLPCM1 > 0)
            {
              Input[input].DPLLPCM1--;
            }
          else Input[input].DPLLPCM1 = 8;
        }
      paramDPLL=1;
      if ((sig_type==1 || sig_type==2 || sig_type==3) && (SR == 6))
        {
          DPLL_PCM = Input[input].DPLLPCM1;
          setDPLLPCM();
        }
      menu_dpll_disp(false);
      savetoEEPROM();
      selectDPLLMillis=millis();
      irdir=0;
      break;
      
      case 2:                                                   // PCM 705.6K
      if (dir==1 || irdir==1)  // CW
        {
          if (Input[input].DPLLPCM2 < 8)
            {
              Input[input].DPLLPCM2++;
            }
          else Input[input].DPLLPCM2 = 0;
        }
      else                                                      // If not CW, then it is CCW
        {
          if (Input[input].DPLLPCM2 > 0)
            {
              Input[input].DPLLPCM2--;
            }
          else Input[input].DPLLPCM2 = 8;
        }
      paramDPLL=2;
      if ((sig_type==1 || sig_type==2 || sig_type==3) && (SR == 7))
        {
          DPLL_PCM = Input[input].DPLLPCM2;
          setDPLLPCM();
        }
      menu_dpll_disp(false);
      savetoEEPROM();
      selectDPLLMillis=millis();
      irdir=0;
      break;

      case 3:                                                   // PCM 384K
      if (dir==1 || irdir==1)  // CW
        {
          if (Input[input].DPLLPCM3 < 8)
            {
              Input[input].DPLLPCM3++;
            }
          else Input[input].DPLLPCM3 = 0;
        }
      else                                                      // If not CW, then it is CCW
        {
          if (Input[input].DPLLPCM3 > 0)
            {
              Input[input].DPLLPCM3--;
            }
          else Input[input].DPLLPCM3 = 8;
        }
      paramDPLL=3;
      if ((sig_type==1 || sig_type==2 || sig_type==3) && (SR == 8))
        {
          DPLL_PCM = Input[input].DPLLPCM3;
          setDPLLPCM();
        }
      menu_dpll_disp(false);
      savetoEEPROM();
      selectDPLLMillis=millis();
      irdir=0;
      break;

      case 4:                                                   // PCM 352.8K
      if (dir==1 || irdir==1)  // CW
        {
          if (Input[input].DPLLPCM4 < 8)
            {
              Input[input].DPLLPCM4++;
            }
          else Input[input].DPLLPCM4 = 0;
        }
      else                                                      // If not CW, then it is CCW
        {
          if (Input[input].DPLLPCM4 > 0)
            {
              Input[input].DPLLPCM4--;
            }
          else Input[input].DPLLPCM4 = 8;
        }
      paramDPLL=4;
      if ((sig_type==1 || sig_type==2 || sig_type==3) && (SR == 9))
        {
          DPLL_PCM = Input[input].DPLLPCM4;
          setDPLLPCM();
        }
      menu_dpll_disp(false);
      savetoEEPROM();
      selectDPLLMillis=millis();
      irdir=0;
      break;

      case 5:                                                   // PCM 192K
      if (dir==1 || irdir==1)  // CW
        {
          if (Input[input].DPLLPCM5 < 8)
            {
              Input[input].DPLLPCM5++;
            }
          else Input[input].DPLLPCM5 = 0;
        }
      else                                                      // If not CW, then it is CCW
        {
          if (Input[input].DPLLPCM5 > 0)
            {
              Input[input].DPLLPCM5--;
            }
          else Input[input].DPLLPCM5 = 8;
        }
      paramDPLL=5;
      if ((sig_type==1 || sig_type==2 || sig_type==3) && (SR == 10))
        {
          DPLL_PCM = Input[input].DPLLPCM5;
          setDPLLPCM();
        }
      menu_dpll_disp(false);
      savetoEEPROM();
      selectDPLLMillis=millis();
      irdir=0;
      break;

      case 6:                                                   // PCM 176.4K
      if (dir==1 || irdir==1)  // CW
        {
          if (Input[input].DPLLPCM6 < 8)
            {
              Input[input].DPLLPCM6++;
            }
          else Input[input].DPLLPCM6 = 0;
        }
      else                                                      // If not CW, then it is CCW
        {
          if (Input[input].DPLLPCM6 > 0)
            {
              Input[input].DPLLPCM6--;
            }
          else Input[input].DPLLPCM6 = 8;
        }
      paramDPLL=6;
      if ((sig_type==1 || sig_type==2 || sig_type==3) && (SR == 11))
        {
          DPLL_PCM = Input[input].DPLLPCM6;
          setDPLLPCM();
        }
      menu_dpll_disp(false);
      savetoEEPROM();
      selectDPLLMillis=millis();
      irdir=0;
      break;

      case 7:                                                   // PCM 96K
      if (dir==1 || irdir==1)  // CW
        {
          if (Input[input].DPLLPCM7 < 8)
            {
              Input[input].DPLLPCM7++;
            }
          else Input[input].DPLLPCM7 = 0;
        }
      else                                                      // If not CW, then it is CCW
        {
          if (Input[input].DPLLPCM7 > 0)
            {
              Input[input].DPLLPCM7--;
            }
          else Input[input].DPLLPCM7 = 8;
        }
      paramDPLL=7;
      if ((sig_type==1 || sig_type==2 || sig_type==3) && (SR == 12))
        {
          DPLL_PCM = Input[input].DPLLPCM7;
          setDPLLPCM();
        }
      menu_dpll_disp(false);
      savetoEEPROM();
      selectDPLLMillis=millis();
      irdir=0;
      break;

      case 8:                                                   // PCM 88.2K
      if (dir==1 || irdir==1)  // CW
        {
          if (Input[input].DPLLPCM8 < 8)
            {
              Input[input].DPLLPCM8++;
            }
          else Input[input].DPLLPCM8 = 0;
        }
      else                                                      // If not CW, then it is CCW
        {
          if (Input[input].DPLLPCM8 > 0)
            {
              Input[input].DPLLPCM8--;
            }
          else Input[input].DPLLPCM8 = 8;
        }
      paramDPLL=8;
      if ((sig_type==1 || sig_type==2 || sig_type==3) && (SR == 13))
        {
          DPLL_PCM = Input[input].DPLLPCM8;
          setDPLLPCM();
        }
      menu_dpll_disp(false);
      savetoEEPROM();
      selectDPLLMillis=millis();
      irdir=0;
      break;

      case 9:                                                   // PCM 48K
      if (dir==1 || irdir==1)  // CW
        {
          if (Input[input].DPLLPCM9 < 8)
            {
              Input[input].DPLLPCM9++;
            }
          else Input[input].DPLLPCM9 = 0;
        }
      else                                                      // If not CW, then it is CCW
        {
          if (Input[input].DPLLPCM9 > 0)
            {
              Input[input].DPLLPCM9--;
            }
          else Input[input].DPLLPCM9 = 8;
        }
      paramDPLL=9;
      if ((sig_type==1 || sig_type==2 || sig_type==3) && (SR == 14))
        {
          DPLL_PCM = Input[input].DPLLPCM9;
          setDPLLPCM();
        }
      menu_dpll_disp(false);
      savetoEEPROM();
      selectDPLLMillis=millis();
      irdir=0;
      break;

      case 10:                                                   // PCM 44.1K
      if (dir==1 || irdir==1)  // CW
        {
          if (Input[input].DPLLPCM10 < 8)
            {
              Input[input].DPLLPCM10++;
            }
          else Input[input].DPLLPCM10 = 0;
        }
      else                                                      // If not CW, then it is CCW
        {
          if (Input[input].DPLLPCM10 > 0)
            {
              Input[input].DPLLPCM10--;
            }
          else Input[input].DPLLPCM10 = 8;
        }
      paramDPLL=10;
      if ((sig_type==1 || sig_type==2 || sig_type==3) && (SR == 15))
        {
          DPLL_PCM = Input[input].DPLLPCM10;
          setDPLLPCM();
        }
      menu_dpll_disp(false);
      savetoEEPROM();
      selectDPLLMillis=millis();
      irdir=0;
      break;

      case 11:                                                   // PCM 32K
      if (dir==1 || irdir==1)  // CW
        {
          if (Input[input].DPLLPCM11 < 8)
            {
              Input[input].DPLLPCM11++;
            }
          else Input[input].DPLLPCM11 = 0;
        }
      else                                                      // If not CW, then it is CCW
        {
          if (Input[input].DPLLPCM11 > 0)
            {
              Input[input].DPLLPCM11--;
            }
          else Input[input].DPLLPCM11 = 8;
        }
      paramDPLL=11;
      if ((sig_type==1 || sig_type==2 || sig_type==3) && (SR == 16))
        {
          DPLL_PCM = Input[input].DPLLPCM11;
          setDPLLPCM();
        }
      menu_dpll_disp(false);
      savetoEEPROM();
      selectDPLLMillis=millis();
      irdir=0;
      if ((input != 0) || (input != 1))
        {
          selectD = 0;
        }
      break;

      case 12:                                                   // DSD 1024
      if (dir==1 || irdir==1)  // CW
        {
          if (Input[input].DPLLDSD1 < 9)
            {
              Input[input].DPLLDSD1++;
            }
          else Input[input].DPLLDSD1 = 0;
        }
      else                                                      // If not CW, then it is CCW
        {
          if (Input[input].DPLLDSD1 > 0)
            {
              Input[input].DPLLDSD1--;
            }
          else Input[input].DPLLDSD1 = 9;
        }
      paramDPLL=12;
      if ((sig_type==0) && (SR == 1))
        {
          DPLL_DSD = Input[input].DPLLDSD1;
          setDPLLDSD();
        }      
      menu_dpll_disp(false);
      SerialUSB.println(Input[input].DPLLDSD1);
      savetoEEPROM();
      selectDPLLMillis=millis();
      irdir=0;
      break;

      case 13:                                                   // DSD 512
      if (dir==1 || irdir==1)  // CW
        {
          if (Input[input].DPLLDSD2 < 9)
            {
              Input[input].DPLLDSD2++;
            }
          else Input[input].DPLLDSD2 = 0;
        }
      else                                                      // If not CW, then it is CCW
        {
          if (Input[input].DPLLDSD2 > 0)
            {
              Input[input].DPLLDSD2--;
            }
          else Input[input].DPLLDSD2 = 9;
        }
      paramDPLL=13;
      if ((sig_type==0) && (SR == 2))
        {
          DPLL_DSD = Input[input].DPLLDSD2;
          setDPLLDSD();
        }   
      menu_dpll_disp(false);
      savetoEEPROM();
      selectDPLLMillis=millis();
      irdir=0;
      break;

      case 14:                                                   // DSD 256
      if (dir==1 || irdir==1)  // CW
        {
          if (Input[input].DPLLDSD3 < 9)
            {
              Input[input].DPLLDSD3++;
            }
          else Input[input].DPLLDSD3 = 0;
        }
      else                                                      // If not CW, then it is CCW
        {
          if (Input[input].DPLLDSD3 > 0)
            {
              Input[input].DPLLDSD3--;
            }
          else Input[input].DPLLDSD3 = 9;
        }
      paramDPLL=14;
      if ((sig_type==0) && (SR == 3))
        {
          DPLL_DSD = Input[input].DPLLDSD3;
          setDPLLDSD();
        }   
      menu_dpll_disp(false);
      savetoEEPROM();
      selectDPLLMillis=millis();
      irdir=0;
      break;

      case 15:                                                   // DSD 128
      if (dir==1 || irdir==1)  // CW
        {
          if (Input[input].DPLLDSD4 < 9)
            {
              Input[input].DPLLDSD4++;
            }
          else Input[input].DPLLDSD4 = 0;
        }
      else                                                      // If not CW, then it is CCW
        {
          if (Input[input].DPLLDSD4 > 0)
            {
              Input[input].DPLLDSD4--;
            }
          else Input[input].DPLLDSD4 = 9;
        }
      paramDPLL=15;
      if ((sig_type==0) && (SR == 4))
        {
          DPLL_DSD = Input[input].DPLLDSD4;
          setDPLLDSD();
        }   
      menu_dpll_disp(false);
      savetoEEPROM();
      selectDPLLMillis=millis();
      irdir=0;
      break;

      case 16:                                                   // DSD 64
      if (dir==1 || irdir==1)  // CW
        {
          if (Input[input].DPLLDSD5 < 9)
            {
              Input[input].DPLLDSD5++;
            }
          else Input[input].DPLLDSD5 = 0;
        }
      else                                                      // If not CW, then it is CCW
        {
          if (Input[input].DPLLDSD5 > 0)
            {
              Input[input].DPLLDSD5--;
            }
          else Input[input].DPLLDSD5 = 9;
        }
      paramDPLL=16;
      if ((sig_type==0) && (SR == 5))
        {
          DPLL_DSD = Input[input].DPLLDSD5;
          setDPLLDSD();
        }   
      menu_dpll_disp(false);
      savetoEEPROM();
      selectDPLLMillis=millis();
      irdir=0;
      selectD = 0;
      break;      
    }
    dir=0;
  }                                                                 // End of "in selectDPLLMode"



// ----------------------------------------------- Rotary Encoder button Code ---------------------------------------------------------------
  buttonPressed();                                                  // Get status of button
  if (key==1)                                                       // Momentary press of the button
    {
      if ((select1Mode == false) && (select2Mode == false) && (selectDPLLMode == false))         // If not in Select mode
        {
          if (poweron == false)                                     // If not powered on, do power on.
          {
            poweron = true;
            digitalWrite(POWERPIN, LOW);
            SerialUSB.println(F("Powering on"));
            setBacklight();            
            DeviceInit();
            setInput();
            displayHelp();                                          // Print help text on serial port
            delay(100);
            key=0;
          }
          else if ((poweron == true) && (preamp == true))           // If powered on and in preamp mode, do source selection.
          {
            if (do_select_inp == 0)
              {
                do_select_inp = 1;
                main_disp(false);
                select1Millis=millis();
              }
              
            else do_select_inp = 0;
            main_disp(false);
            key=0;
          }
          else if ((poweron == true) && (preamp == false))          // If powered on but not in preamp mode, do filter selection.
          {
            do_select_inp = 0;
            if (do_select_filter == 0)
              {
                do_select_filter = 1;
                select1Millis=millis();
              }
              
            else do_select_filter = 0;
            main_disp(false);
            key=0;
          }
        }
      else if (select1Mode == true)                                 // If in Select mode
        {
          select1++;
          SerialUSB.println(select1);
          switch(select1%(MAXPARAM))
            {  
              case FIR:
              param=FIR;
              menu1_disp(false);
              select1Millis=millis();
              key=0;
              break;
      
              case IIRV:
              param=IIRV;
              menu1_disp(false);
              select1Millis=millis();
              key=0;
              break;

              case LCK:
              param=LCK;
              menu1_disp(false);
              select1Millis=millis();
              key=0;
              break;              

              case DTH:
              param=DTH;
              menu1_disp(false);
              select1Millis=millis();
              key=0;
              break;

              case JTR:
              param=JTR;
              menu1_disp(false);
              select1Millis=millis();
              key=0;
              break;

              case OSF:
              param=OSF;
              menu1_disp(false);
              select1Millis=millis();
              key=0;
              break;

              case DMF:
              param=DMF;
              menu1_disp(false);
              select1Millis=millis();
              key=0;
              break;

              case DPL:
              param=DPL;
              menu1_disp(false);
              select1Millis=millis();
              key=0;
              break;
           
           }
      }
      else if (select2Mode == true)                                       // If in Select mode
        {
          select2++;
          SerialUSB.print(F("Select2: ")); SerialUSB.println(select2);
          switch(select2%(MAXPARAM2))
            {  
              case PRE:
              setting=PRE;
              settings_disp(false);
              select2Millis=millis();
              key=0;
              break;
      
              case PRE_VOL:
              setting=PRE_VOL;
              settings_disp(false);
              select2Millis=millis();
              key=0;
              break;

              case BLGHT:
              setting=BLGHT;
              settings_disp(false);
              select2Millis=millis();
              key=0;
              break;              
           }
       }

      else if (selectDPLLMode == true)                                       // If in Select DPLL mode
        {
          if (selectD == 0)
            {
              if ((input == 0) || (input == 1))
                {
                  selectD = 1;
                }
              else selectD = 5;
            }
          else if (selectD < 16)
                {
                  selectD++;
                }
                else selectD = 0;
            
          SerialUSB.print(F("SelectD: ")); SerialUSB.println(selectD);
          switch(selectD)
            {  
              case 1:
              paramDPLL=1;
              menu_dpll_disp(false);
              selectDPLLMillis=millis();
              key=0;
              break;
      
              case 2:
              paramDPLL=2;
              menu_dpll_disp(false);
              selectDPLLMillis=millis();
              key=0;
              break;

              case 3:
              paramDPLL=3;
              menu_dpll_disp(false);
              selectDPLLMillis=millis();
              key=0;
              break;

              case 4:
              paramDPLL=4;
              menu_dpll_disp(false);
              selectDPLLMillis=millis();
              key=0;
              break;
      
              case 5:
              paramDPLL=5;
              menu_dpll_disp(false);
              selectDPLLMillis=millis();
              key=0;
              break;

              case 6:
              paramDPLL=6;
              menu_dpll_disp(false);
              selectDPLLMillis=millis();
              key=0;
              break;
              
              case 7:
              paramDPLL=7;
              menu_dpll_disp(false);
              selectDPLLMillis=millis();
              key=0;
              break;
      
              case 8:
              paramDPLL=8;
              menu_dpll_disp(false);
              selectDPLLMillis=millis();
              key=0;
              break;

              case 9:
              paramDPLL=9;
              menu_dpll_disp(false);
              selectDPLLMillis=millis();
              key=0;
              break;

              case 10:
              paramDPLL=10;
              menu_dpll_disp(false);
              selectDPLLMillis=millis();
              key=0;
              break;
      
              case 11:
              paramDPLL=11;
              menu_dpll_disp(false);
              selectDPLLMillis=millis();
              key=0;
              //if ((input != 0) || (input != 1))
                //{
                  //selectD = 0;
                //}
              break;

              case 12:
              paramDPLL=12;
              menu_dpll_disp(false);
              selectDPLLMillis=millis();
              key=0;
              break;  

              case 13:
              paramDPLL=13;
              menu_dpll_disp(false);
              selectDPLLMillis=millis();
              key=0;
              break;

              case 14:
              paramDPLL=14;
              menu_dpll_disp(false);
              selectDPLLMillis=millis();
              key=0;
              break;
      
              case 15:
              paramDPLL=15;
              menu_dpll_disp(false);
              selectDPLLMillis=millis();
              key=0;
              break;

              case 16:
              paramDPLL=16;
              menu_dpll_disp(false);
              selectDPLLMillis=millis();
              key=0;
              break;               
           }
       }

   }
 
  else if (key==2)                                                        // Long press of the button
    {
      if (select1Mode == false)
        {
          SerialUSB.println(F("Go into menu mode."));
          select1Mode = true;
          select1Millis=millis();                                         // Start being-in-select-mode timer
          param=1;
          menu1_disp(true);
          key=0;          
        }
      else if (select1Mode == true)
        {
          SerialUSB.println(F("Go into settings menu mode."));
          select1Mode = false;
          select2Mode = true;
          select2Millis=millis();                                         // Start being-in-select-mode timer
          setting=1;
          param=1;
          settings_disp(true);
          key=0;          
        }
    }
  
  else if (key==3)                                                        // Very long press of the button
    {
      SerialUSB.println(F("Power Off"));
      poweron = false;
      //digitalWrite(MUTE, LOW);
      delay(1000);
      myGLCD.clrScr();
      digitalWrite(POWERPIN, HIGH);
      SerialUSB.println(F("Powering off"));
      backlight=0;
      setBacklight();
      //PowerOff_disp();
      key=0;
    }

  
// End of rotary encoder button code

// ----------------------------------------------- IR Receiver Code ---------------------------------------------------------------

  if (irrecv.decode(&results)) {                                          // If IR code is received, get the IR code.
    //SerialUSB.println(results.value);
    if (results.value == POWER_CODE)                                      // What to do if the Power button is pressed
      {
        if (poweron == false)                                             // If not powered on, do power on.
          {
            poweron = true;
            digitalWrite(POWERPIN, LOW);
            SerialUSB.println(F("Powering on"));
            setBacklight();
            DeviceInit();
            //setInput();
            displayHelp();                                                // Print help text on serial port
            prev_result = 0;
          }
        else if (poweron == true)                                         // If powered on, power off.
          {
            poweron = false;
            digitalWrite(POWERPIN, HIGH);
            SerialUSB.println(F("Powering off"));
            myGLCD.clrScr();
            backlight=0;
            setBacklight();
            //delay(2000);
            //digitalWrite(MUTE, HIGH);
            prev_result = 0;
          }
      } 

    else if (results.value == SELECT_CODE)                               // What to do if the Select button is pressed
      {
        if (poweron == true) 
          {
            key = 1;                                                     // Set the variable "key" to 1, so same as pressing the rot enc.
          }
        prev_result = SELECT_CODE;
      }  

    else if (results.value == LEFT_CODE)                                 // What to do if the Left button is pressed
      {
        //SerialUSB.write("Left\n");
        if (poweron == true)
          {
            irdir = 2;    
          }
        prev_result = 0;
      } 
    
    else if (results.value == RIGHT_CODE)                                 // What to do if the Right button is pressed
      {
        //SerialUSB.write("Right\n");
        if (poweron == true)
          { 
            irdir = 1;  
          }
        prev_result = 0;
      } 

    else if (results.value == MUTE_CODE)                                  // What to do if the Mute button is pressed
      {
      //SerialUSB.write("Mute\n");
        if ((poweron == true) && (mute == false))                         // If powered on and not muted, do mute
          {
            mute = true;
            setMute();
            main_disp(false);
          }
        else if ((poweron == true) && (mute == true))                     // If powered on and mute, do unmute
          {
            mute = false;
            setMute();
            main_disp(false);
          }
        prev_result = 0;
      }     
    
    else if ((results.value == SOURCE1_CODE) && (poweron == true))        // What to do if the Source 1 button is pressed
      {
        input = 0;
        setInput();
        prev_result = 0;
        main_disp(false);
      }
        
    else if ((results.value == SOURCE2_CODE) && (poweron == true))        // What to do if the Source 2 button is pressed
      {
        input = 1;
        setInput();
        prev_result = 0;
        main_disp(false);
      }

    else if ((results.value == SOURCE3_CODE) && (poweron == true))        // What to do if the Source 3 button is pressed
      {
        input = 2;
        setInput();
        prev_result = 0;
        main_disp(false);
      }

    else if ((results.value == SOURCE4_CODE) && (poweron == true))        // What to do if the Source 4 button is pressed
      {
        input = 3;
        setInput();
        prev_result = 0;
        main_disp(false);
      }
    
    else if ((results.value == SOURCE5_CODE) && (poweron == true))        // What to do if the Source 5 button is pressed
      {
        input = 4;
        setInput();
        prev_result = 0;
        main_disp(false);
      }

    else if ((results.value == SOURCE6_CODE) && (poweron == true))        // What to do if the Source 6 button is pressed
      {
        input = 5;
        setInput();
        prev_result = 0;
        main_disp(false);
      }

    else if ((results.value == VOLUP_CODE) && (poweron == true))          // What to do if the Volume Up button is pressed
      {  
        if (poweron)
          {
            if ((volume > 0) && (preamp == true))                         // If in preamp mode, change volume
              {
                volume=volume-2;
                setVol();
                main_disp(false);
                irdir=0;
              }      
            prev_result = VOLUP_CODE;
          }
      }

    else if ((results.value == VOLDOWN_CODE) && (poweron == true))        // What to do if the Volume Down button is pressed
      {    
        if (poweron)
          {
            if ((volume < 198) && (preamp == true))                       // If in preamp mode, change volume
              {
                volume=volume+2;
                setVol();
                main_disp(false);
                irdir=0;       
              } 
            prev_result = VOLDOWN_CODE;
          }
      }

    else if ((results.value == SETTINGS_CODE) && (poweron == true))       // What to do if the Settings button is pressed
      {    
        if (poweron)
          {
            key = 2;
            prev_result = SETTINGS_CODE;
          }
      }

    else if ((results.value == GLOBAL_SET_CODE) && (poweron == true))       // What to do if the Settings button is pressed
      {    
        if (poweron)
          {
            SerialUSB.println(F("Go into global settings menu mode."));
            select1Mode = false;
            select2Mode = true;
            select2Millis=millis();                                         // Start being-in-select-mode timer
            setting=1;
            param=1;
            settings_disp(true);
            prev_result = GLOBAL_SET_CODE;
          }
      }
    

    else if (results.value == 0xFFFFFFFF)                                 // What to do if the "repeat" code (FFFFFFFF) is detected
      {
        switch(prev_result)
          {
          case VOLUP_CODE:                                                // If the previously pressed button was VOLUP_CODE
            if (poweron)
              {
                if ((volume > 0) && (preamp == true))                     // If in preamp mode, change volume
                  {
                    volume=volume-2;
                    setVol();
                    main_disp(false);
                    irdir=0;
                    prev_result == 0;
                  }      
              }
            break;
         
          case VOLDOWN_CODE:                                               // If the previously pressed button was VOLDOWN_CODE
            SerialUSB.println(F("Volume Down"));
            if (poweron)
              {
                if ((volume < 199) && (preamp == true))                    // If in preamp mode, change volume
                  {
                    volume=volume+2;
                    setVol();
                    main_disp(false);
                    irdir=0;
                    prev_result == 0;
                  }       
              }
            break;
    
          /*
          case SELECT_CODE:                                                 // If the previously pressed button was SELECT_CODE
            if (poweron)
              {
                key = 2;                                                    // Long press of key
                prev_result == 0;
              }
            break;
          */
          }
      }
    
    else 
      {
        SerialUSB.print(F("unexpected value: "));                               // If the IR code is not recognized, output it to Serial
        SerialUSB.println(results.value, HEX);
        prev_result = 0;
      }
    irrecv.resume();                                                         // Resume decoding (necessary!)
  }

// ----------------------------------------------- Come out of Source Selection mode if it times out ---------------------------------------------------------------

  if((selectDPLLMode==false) && (do_select_inp&&millis()-select1Millis > INTERVAL_SELECT*1000))
    {
      do_select_inp = 0;
      //select1 = 1;
      main_disp(false);
      SerialUSB.println(F("Came out of Source Selection mode"));
    } 

// ----------------------------------------------- Come out of Select1Mode if it times out ---------------------------------------------------------------

  if((select1Mode==true) && (selectDPLLMode==false) && (select1Mode&&millis()-select1Millis > INTERVAL_SELECT*1000))
    {
      select1Mode=false;                        // No longer in select mode
      do_select_inp = 0;
      param = 1;
      setting = 1;
      select1 = 1;
      main_disp(true);
      SerialUSB.println(F("Came out of menu mode"));
    } 

// ----------------------------------------------- Come out of SelectDPLLMode if it times out ---------------------------------------------------------------

  if((selectDPLLMode==true) && (selectDPLLMode&&millis()-selectDPLLMillis > INTERVAL_SELECT*1000))
    {
      selectDPLLMode=false;                        // No longer in select DPLL mode
      do_select_inp = 0;
      param = 1;
      setting = 1;
      select1 = 1;
      main_disp(true);
      SerialUSB.println(F("Came out of DPLL menu mode"));
    } 

// ----------------------------------------------- Come out of Select2Mode if it times out ---------------------------------------------------------------

  if((select2Mode==true) && (selectDPLLMode==false) && (select2Mode&&millis()-select2Millis > INTERVAL_SELECT*1000))
    {
      select2Mode=false;                        // No longer in select mode
      select2 = PRE;
      param = 1;
      setting = 1;
      select1 = 1;
      main_disp(true);
      SerialUSB.println(F("Came out of settings menu mode"));
    } 

// ----------------------------------------------- Serial port prompt code ---------------------------------------------------------------

char command = getCommand();
switch (command)
  {
    case 'a':
      input=0;
      setInput();
      main_disp(false);
      break;
    case 'b':
      input=1;
      setInput();
      main_disp(false);
      break;
    case 'c':
      input=2;
      setInput();
      main_disp(false);
      break;
    case 'd':
      IIR=0;
      setIIR();
      main_disp(false);
      break;
    case 'e':
      IIR=1;
      setIIR();
      main_disp(false);
      break;
    case 'f':
      IIR=2;
      setIIR();
      main_disp(false);
      break;
    case 'g':
      IIR=3;
      setIIR();
      main_disp(false);
      break;
    case 'h':
      mute=0;
      setMute();
      main_disp(false);
      break;
    case 'i':
      mute=1;
      setMute();
      main_disp(false);
      break;
    case 'j':
      getLockStatus();
      getSR();
      main_disp(false);
      break;
    case '0':
      filter=0;
      setFilter();
      main_disp(false);
      break;
    case '1':
      filter=1;
      setFilter();
      main_disp(false);
      break;
    case '2':
      filter=2;
      setFilter();
      main_disp(false);
      break;
    case '3':
      filter=3;
      setFilter();
      main_disp(false);
      break;
    case '4':
      filter=4;
      setFilter();
      main_disp(false);
      break;
    case '5':
      filter=5;
      setFilter();
      main_disp(false);
      break;
    case '6':
      filter=6;
      setFilter();
      main_disp(false);
      break;
    case 'y':
      PrintDebugReg(dac);
      break;
    case 'z':
      displayHelp();
      break;
    case '!':
      initEEPROM();
      break;
    case '@':
      DeviceInit();
      break;
    case '^':
      testEEPROM();
      break;
    default:
      break;
  }

}

// ----------------------------------------------- End of main loop()---------------------------------------------------------------

// ----------------------------------------------  Function that selects the input and applies input-specific settings.
void setInput()
{
  if (input == 0)                       // Select I2S/DSD input 1 with Auto detection
    {
      mcp.digitalWrite(7, HIGH);        // Select I2S on the Solid State Sidecar
      digitalWrite(SIDECARPIN, HIGH);   // Select I2S on the Solid State Sidecar (if not using MCP chip)
      mcp.digitalWrite(6, LOW);         // Select I2S input 1 on SSS.
      ChangeBit(dac, 0x01, 3, 1);       // Enable auto signal detection
      ChangeBit(dac, 0x01, 2, 1);       // Enable auto signal detection
      ChangeBit(dac, 0x0B, 7, 0);
      ChangeBit(dac, 0x0B, 6, 0);
      ChangeBit(dac, 0x0B, 5, 0);
      ChangeBit(dac, 0x0B, 4, 0);
      filter = Input[0].filter_val; setFilter();
      IIR = Input[0].IIR_val; setIIR();
      LockSpeed = Input[0].lockspeed_val; setLockSpeed();
      Dither = Input[0].dither_val; setDither();
      JitterElim = Input[0].jitter_elim_val; setJitterElim();
      bypass_osf = Input[0].bypass_osf_val; 
      deemph_bypass = Input[0].deemph_bypass_val; setOSF();
      auto_deemph = Input[0].auto_deemph_val;
      deemph_sel = Input[0].deemph_sel_val; setDeemphasis();
    }
  else if (input == 1)                  // Select I2S/DSD input 2 with Auto detection
    {
      mcp.digitalWrite(7, HIGH);        // Select I2S on the Solid State Sidecar
      digitalWrite(SIDECARPIN, HIGH);   // Select I2S on the Solid State Sidecar (if not using MCP chip)
      mcp.digitalWrite(6, HIGH);        // Select I2S input 2 on SSS.
      ChangeBit(dac, 0x01, 3, 1);       // Enable auto signal detection
      ChangeBit(dac, 0x01, 2, 1);       // Enable auto signal detection
      ChangeBit(dac, 0x0B, 7, 0);
      ChangeBit(dac, 0x0B, 6, 0);
      ChangeBit(dac, 0x0B, 5, 0);
      ChangeBit(dac, 0x0B, 4, 0);
      filter = Input[1].filter_val; setFilter();
      IIR = Input[1].IIR_val; setIIR();
      LockSpeed = Input[1].lockspeed_val; setLockSpeed();
      Dither = Input[1].dither_val; setDither();
      JitterElim = Input[1].jitter_elim_val; setJitterElim();
      bypass_osf = Input[1].bypass_osf_val; 
      deemph_bypass = Input[1].deemph_bypass_val; setOSF();
      auto_deemph = Input[1].auto_deemph_val;
      deemph_sel = Input[1].deemph_sel_val; setDeemphasis();
    }
  else if (input == 2)                  // Select s/pdif from DATA1 pin
    {
      mcp.digitalWrite(7, LOW);         // Select SPDIF on the Solid State Sidecar
      digitalWrite(SIDECARPIN, LOW);    // Select SPDIF on the Solid State Sidecar (if not using MCP chip)
      ChangeBit(dac, 0x01, 3, 0);       // Disable auto signal detection
      ChangeBit(dac, 0x01, 2, 0);       // Disable auto signal detection
      ChangeBit(dac, 0x01, 1, 0);       // Set input signal type to spdif
      ChangeBit(dac, 0x01, 0, 1);       // Set input signal type to spdif
      ChangeBit(dac, 0x0B, 7, 0);       // Select s/pdif from DATA1 pin
      ChangeBit(dac, 0x0B, 6, 0);       // Select s/pdif from DATA1 pin
      ChangeBit(dac, 0x0B, 5, 0);       // Select s/pdif from DATA1 pin
      ChangeBit(dac, 0x0B, 4, 1);       // Select s/pdif from DATA1 pin
      filter = Input[2].filter_val; setFilter();
      IIR = Input[2].IIR_val; setIIR();
      LockSpeed = Input[2].lockspeed_val; setLockSpeed();
      Dither = Input[2].dither_val; setDither();
      JitterElim = Input[2].jitter_elim_val; setJitterElim();
      bypass_osf = Input[2].bypass_osf_val; 
      deemph_bypass = Input[2].deemph_bypass_val; setOSF();
      auto_deemph = Input[2].auto_deemph_val;
      deemph_sel = Input[2].deemph_sel_val; setDeemphasis(); 
    }
  else if (input == 3)                  // Select s/pdif from DATA2 pin
    {
      mcp.digitalWrite(7, LOW);         // Select SPDIF on the Solid State Sidecar
      digitalWrite(SIDECARPIN, LOW);    // Select SPDIF on the Solid State Sidecar (if not using MCP chip)
      ChangeBit(dac, 0x01, 3, 0);       // Disable auto signal detection
      ChangeBit(dac, 0x01, 2, 0);       // Disable auto signal detection
      ChangeBit(dac, 0x01, 1, 0);       // Set input signal type to spdif
      ChangeBit(dac, 0x01, 0, 1);       // Set input signal type to spdif
      ChangeBit(dac, 0x0B, 7, 0);       // Select s/pdif from DATA2 pin
      ChangeBit(dac, 0x0B, 6, 0);       // Select s/pdif from DATA2 pin
      ChangeBit(dac, 0x0B, 5, 1);       // Select s/pdif from DATA2 pin
      ChangeBit(dac, 0x0B, 4, 0);       // Select s/pdif from DATA2 pin
      filter = Input[3].filter_val; setFilter();
      IIR = Input[3].IIR_val; setIIR();
      LockSpeed = Input[3].lockspeed_val; setLockSpeed();
      Dither = Input[3].dither_val; setDither();
      JitterElim = Input[3].jitter_elim_val; setJitterElim();
      bypass_osf = Input[3].bypass_osf_val; 
      deemph_bypass = Input[3].deemph_bypass_val; setOSF();
      auto_deemph = Input[3].auto_deemph_val;
      deemph_sel = Input[3].deemph_sel_val; setDeemphasis();      
    }
  else if (input == 4)                  // Select s/pdif from DATA3 pin
    {
      mcp.digitalWrite(7, LOW);         // Select SPDIF on the Solid State Sidecar
      digitalWrite(SIDECARPIN, LOW);    // Select SPDIF on the Solid State Sidecar (if not using MCP chip)
      ChangeBit(dac, 0x01, 3, 0);       // Disable auto signal detection
      ChangeBit(dac, 0x01, 2, 0);       // Disable auto signal detection
      ChangeBit(dac, 0x01, 1, 0);       // Set input signal type to spdif
      ChangeBit(dac, 0x01, 0, 1);       // Set input signal type to spdif
      ChangeBit(dac, 0x0B, 7, 0);       // Select s/pdif from DATA3 pin
      ChangeBit(dac, 0x0B, 6, 0);       // Select s/pdif from DATA3 pin
      ChangeBit(dac, 0x0B, 5, 1);       // Select s/pdif from DATA3 pin
      ChangeBit(dac, 0x0B, 4, 1);       // Select s/pdif from DATA3 pin
      filter = Input[4].filter_val; setFilter();
      IIR = Input[4].IIR_val; setIIR();
      LockSpeed = Input[4].lockspeed_val; setLockSpeed();
      Dither = Input[4].dither_val; setDither();
      JitterElim = Input[4].jitter_elim_val; setJitterElim();
      bypass_osf = Input[4].bypass_osf_val; 
      deemph_bypass = Input[4].deemph_bypass_val; setOSF();
      auto_deemph = Input[4].auto_deemph_val;
      deemph_sel = Input[4].deemph_sel_val; setDeemphasis();        
    }
  else if (input == 5)                  // Select s/pdif from DATA4 pin
    {
      mcp.digitalWrite(7, LOW);         // Select SPDIF on the Solid State Sidecar
      digitalWrite(SIDECARPIN, LOW);    // Select SPDIF on the Solid State Sidecar (if not using MCP chip)
      ChangeBit(dac, 0x01, 3, 0);       // Disable auto signal detection
      ChangeBit(dac, 0x01, 2, 0);       // Disable auto signal detection
      ChangeBit(dac, 0x01, 1, 0);       // Set input signal type to spdif
      ChangeBit(dac, 0x01, 0, 1);       // Set input signal type to spdif
      ChangeBit(dac, 0x0B, 7, 0);       // Select s/pdif from DATA4 pin
      ChangeBit(dac, 0x0B, 6, 1);       // Select s/pdif from DATA4 pin
      ChangeBit(dac, 0x0B, 5, 0);       // Select s/pdif from DATA4 pin
      ChangeBit(dac, 0x0B, 4, 0);       // Select s/pdif from DATA4 pin
      filter = Input[5].filter_val; setFilter();
      IIR = Input[5].IIR_val; setIIR();
      LockSpeed = Input[5].lockspeed_val; setLockSpeed();
      Dither = Input[5].dither_val; setDither();
      JitterElim = Input[5].jitter_elim_val; setJitterElim();
      bypass_osf = Input[5].bypass_osf_val; 
      deemph_bypass = Input[5].deemph_bypass_val; setOSF();
      auto_deemph = Input[5].auto_deemph_val;
      deemph_sel = Input[5].deemph_sel_val; setDeemphasis();      
    }
  SerialUSB.print(F("Input set to: ")); SerialUSB.println(input);
}

// -----------------------------  Function that sets the volume
void setVol()
{
  SerialUSB.print(F("Volume set to: ")); SerialUSB.println(volume);
  WriteRegister(dac, 0x10, volume);
  WriteRegister(dac, 0x11, volume);  
  WriteRegister(dac, 0x12, volume);  
  WriteRegister(dac, 0x13, volume);
  WriteRegister(dac, 0x14, volume);
  WriteRegister(dac, 0x15, volume);
  WriteRegister(dac, 0x16, volume);
  WriteRegister(dac, 0x17, volume);
}

// -----------------------------  Function that sets the filter
/* 0 = Fast Roll-off, Linear Phase Filter
 * 1 = Slow Roll-off, Linear Phase Filter 
 * 2 = Fast Roll-off, Minimum Phase Filter
 * 3 = Slow Roll-off, Minimum Phase
 * 4 = Apodizing, Fast Roll-off, Linear Phase 
 * 5 = Hybrid, Fast Roll-off, Minimum Phase 
 * 6 = Brickwall 
 */
void setFilter()
{
  if ((filter >= 0) && (filter <= 6))
    {
      if (filter == 0)
        {
          ChangeBit(dac, 0x07, 7, 0);
          ChangeBit(dac, 0x07, 6, 0);
          ChangeBit(dac, 0x07, 5, 0);
        }
      else if (filter == 1)
        {
          ChangeBit(dac, 0x07, 7, 0);
          ChangeBit(dac, 0x07, 6, 0);
          ChangeBit(dac, 0x07, 5, 1);      
        }
      else if (filter == 2)
        {
          ChangeBit(dac, 0x07, 7, 0);
          ChangeBit(dac, 0x07, 6, 1);
          ChangeBit(dac, 0x07, 5, 0);      
        }
      else if (filter == 3)
        {
          ChangeBit(dac, 0x07, 7, 0);
          ChangeBit(dac, 0x07, 6, 1);
          ChangeBit(dac, 0x07, 5, 1);      
        }
      else if (filter == 4)
        {
          ChangeBit(dac, 0x07, 7, 1);
          ChangeBit(dac, 0x07, 6, 0);
          ChangeBit(dac, 0x07, 5, 0);      
        }
      else if (filter == 5)
        {
          ChangeBit(dac, 0x07, 7, 1);
          ChangeBit(dac, 0x07, 6, 1);
          ChangeBit(dac, 0x07, 5, 0);      
        }
      else if (filter == 6)
        {
          ChangeBit(dac, 0x07, 7, 1);
          ChangeBit(dac, 0x07, 6, 1);
          ChangeBit(dac, 0x07, 5, 1);      
        }
      SerialUSB.print(F("Filter set to: ")); SerialUSB.println(filter);
    }
  else
    {
      SerialUSB.print(F("Invalid Filter Number (")); SerialUSB.print(filter); SerialUSB.println(F(") re-initializing EEPROM and rebooting."));
      delay(200);
      initEEPROM();
      delay(200);
      initEEPROM();
      delay(200);
      DeviceInit();      
    }
}

// -----------------------------  Function that sets the IIR bandwidth
/* 0 = 47.44KHz
 * 1 = 50KHz 
 * 2 = 60KHz
 * 3 = 70KHz
 */
void setIIR()
{
  if (IIR == 0)
    {
      ChangeBit(dac, 0x07, 2, 0);
      ChangeBit(dac, 0x07, 1, 0);
    }
  else if (IIR == 1)
    {
      ChangeBit(dac, 0x07, 2, 0);
      ChangeBit(dac, 0x07, 1, 1);
    }
  else if (IIR == 2)
    {
      ChangeBit(dac, 0x07, 2, 1);
      ChangeBit(dac, 0x07, 1, 0);
    }
  else if (IIR == 3)
    {
      ChangeBit(dac, 0x07, 2, 1);
      ChangeBit(dac, 0x07, 1, 1);
    }
  SerialUSB.print(F("IIR bandwidth set to: ")); SerialUSB.println(IIR);
}

// -----------------------------  Function to mute the DAC
/* 0 = not mute
 * 1 = mute
 */
void setMute()
{
  if (mute == 0)
    {
      ChangeBit(dac, 0x07, 0, 0);
      SerialUSB.println(F("DAC unmuted"));
    }
  else if (mute == 1)
    {
      ChangeBit(dac, 0x07, 0, 1);
      SerialUSB.println(F("DAC muted"));
    }
}

// -----------------------------  Function to set the DPLL for I2S / SPDIF
/* 0 = DPLL Off
 * 1 = Lowest (1)
 * 2 = Low (3)
 * 3 = Default (5)
 * 4 = Med-Low (7)
 * 5 = Medium (9)
 * 6 = High (11)
 * 7 = Higher (13)
 * 8 = Highest (15)
 */
void setDPLLPCM()
{
  if (DPLL_PCM == 0)
    {
      ChangeBit(dac, 0x0C, 7, 0);
      ChangeBit(dac, 0x0C, 6, 0);
      ChangeBit(dac, 0x0C, 5, 0);
      ChangeBit(dac, 0x0C, 4, 0);            
    }
  else if (DPLL_PCM == 1)
    {
      ChangeBit(dac, 0x0C, 7, 0);
      ChangeBit(dac, 0x0C, 6, 0);
      ChangeBit(dac, 0x0C, 5, 0);
      ChangeBit(dac, 0x0C, 4, 1); 
    }
  else if (DPLL_PCM == 2)
    {
      ChangeBit(dac, 0x0C, 7, 0);
      ChangeBit(dac, 0x0C, 6, 0);
      ChangeBit(dac, 0x0C, 5, 1);
      ChangeBit(dac, 0x0C, 4, 1); 
    }
  else if (DPLL_PCM == 3)
    {
      ChangeBit(dac, 0x0C, 7, 0);
      ChangeBit(dac, 0x0C, 6, 1);
      ChangeBit(dac, 0x0C, 5, 0);
      ChangeBit(dac, 0x0C, 4, 1); 
    }
  else if (DPLL_PCM == 4)
    {
      ChangeBit(dac, 0x0C, 7, 0);
      ChangeBit(dac, 0x0C, 6, 1);
      ChangeBit(dac, 0x0C, 5, 1);
      ChangeBit(dac, 0x0C, 4, 1); 
    }
  else if (DPLL_PCM == 5)
    {
      ChangeBit(dac, 0x0C, 7, 1);
      ChangeBit(dac, 0x0C, 6, 0);
      ChangeBit(dac, 0x0C, 5, 0);
      ChangeBit(dac, 0x0C, 4, 1); 
    }
  else if (DPLL_PCM == 6)
    {
      ChangeBit(dac, 0x0C, 7, 1);
      ChangeBit(dac, 0x0C, 6, 0);
      ChangeBit(dac, 0x0C, 5, 1);
      ChangeBit(dac, 0x0C, 4, 1); 
    }
  else if (DPLL_PCM == 7)
    {
      ChangeBit(dac, 0x0C, 7, 1);
      ChangeBit(dac, 0x0C, 6, 1);
      ChangeBit(dac, 0x0C, 5, 0);
      ChangeBit(dac, 0x0C, 4, 1); 
    } 
  else if (DPLL_PCM == 8)
    {
      ChangeBit(dac, 0x0C, 7, 1);
      ChangeBit(dac, 0x0C, 6, 1);
      ChangeBit(dac, 0x0C, 5, 1);
      ChangeBit(dac, 0x0C, 4, 1); 
    }
}

// -----------------------------  Function to set the DPLL for DSD
/* 0 = DPLL Off
 * 1 = Lowest (1)
 * 2 = Low (3)
 * 3 = Med-Low (5)
 * 4 = Medium (7)
 * 5 = Med-High (9)
 * 6 = Default (10)
 * 7 = High (11)
 * 8 = Higher (13)
 * 9 = Highest (15)
 */

void setDPLLDSD()
{
  if (DPLL_DSD == 0)
    {
      ChangeBit(dac, 0x0C, 3, 0);
      ChangeBit(dac, 0x0C, 2, 0);
      ChangeBit(dac, 0x0C, 1, 0);
      ChangeBit(dac, 0x0C, 0, 0);            
    }
  else if (DPLL_DSD == 1)
    {
      ChangeBit(dac, 0x0C, 3, 0);
      ChangeBit(dac, 0x0C, 2, 0);
      ChangeBit(dac, 0x0C, 1, 0);
      ChangeBit(dac, 0x0C, 0, 1); 
    }
  else if (DPLL_DSD == 2)
    {
      ChangeBit(dac, 0x0C, 3, 0);
      ChangeBit(dac, 0x0C, 2, 0);
      ChangeBit(dac, 0x0C, 1, 1);
      ChangeBit(dac, 0x0C, 0, 1); 
    }
  else if (DPLL_DSD == 3)
    {
      ChangeBit(dac, 0x0C, 3, 0);
      ChangeBit(dac, 0x0C, 2, 1);
      ChangeBit(dac, 0x0C, 1, 0);
      ChangeBit(dac, 0x0C, 0, 1); 
    }
  else if (DPLL_DSD == 4)
    {
      ChangeBit(dac, 0x0C, 3, 0);
      ChangeBit(dac, 0x0C, 2, 1);
      ChangeBit(dac, 0x0C, 1, 1);
      ChangeBit(dac, 0x0C, 0, 1); 
    }
  else if (DPLL_DSD == 5)
    {
      ChangeBit(dac, 0x0C, 3, 1);
      ChangeBit(dac, 0x0C, 2, 0);
      ChangeBit(dac, 0x0C, 1, 0);
      ChangeBit(dac, 0x0C, 0, 1); 
    }
  else if (DPLL_DSD == 6)
    {
      ChangeBit(dac, 0x0C, 3, 1);
      ChangeBit(dac, 0x0C, 2, 0);
      ChangeBit(dac, 0x0C, 1, 1);
      ChangeBit(dac, 0x0C, 0, 0); 
    }
  else if (DPLL_DSD == 7)
    {
      ChangeBit(dac, 0x0C, 3, 1);
      ChangeBit(dac, 0x0C, 2, 0);
      ChangeBit(dac, 0x0C, 1, 1);
      ChangeBit(dac, 0x0C, 0, 1); 
    }
  else if (DPLL_DSD == 8)
    {
      ChangeBit(dac, 0x0C, 3, 1);
      ChangeBit(dac, 0x0C, 2, 1);
      ChangeBit(dac, 0x0C, 1, 0);
      ChangeBit(dac, 0x0C, 0, 1); 
    } 
  else if (DPLL_DSD == 9)
    {
      ChangeBit(dac, 0x0C, 3, 1);
      ChangeBit(dac, 0x0C, 2, 1);
      ChangeBit(dac, 0x0C, 1, 1);
      ChangeBit(dac, 0x0C, 0, 1); 
    }
}

// -----------------------------  Function to set the Lock speed
/* 0 = 16384 FSL edges
 * 1 = 8192 FSL edges (1)
 * 2 = 4096 FSL edges (3)
 * 3 = 2048 FSL edges (7)
 * 4 = 1024 FSL edges (15)
 */
void setLockSpeed()
{
  if (LockSpeed == 0)
    {
      ChangeBit(dac, 0x0A, 3, 0);
      ChangeBit(dac, 0x0A, 2, 0);
      ChangeBit(dac, 0x0A, 1, 0);
      ChangeBit(dac, 0x0A, 0, 1);            
    }
  else if (LockSpeed == 1)
    {
      ChangeBit(dac, 0x0A, 3, 0);
      ChangeBit(dac, 0x0A, 2, 0);
      ChangeBit(dac, 0x0A, 1, 0);
      ChangeBit(dac, 0x0A, 0, 1); 
    }
  else if (LockSpeed == 2)
    {
      ChangeBit(dac, 0x0A, 3, 0);
      ChangeBit(dac, 0x0A, 2, 0);
      ChangeBit(dac, 0x0A, 1, 1);
      ChangeBit(dac, 0x0A, 0, 1); 
    }
  else if (LockSpeed == 3)
    {
      ChangeBit(dac, 0x0A, 3, 0);
      ChangeBit(dac, 0x0A, 2, 1);
      ChangeBit(dac, 0x0A, 1, 1);
      ChangeBit(dac, 0x0A, 0, 1); 
    }
  else if (LockSpeed == 4)
    {
      ChangeBit(dac, 0x10, 3, 1);
      ChangeBit(dac, 0x10, 2, 1);
      ChangeBit(dac, 0x10, 1, 1);
      ChangeBit(dac, 0x10, 0, 1); 
    }
}

// -----------------------------  Function to enable or disable Dithering
/* 0 = enable
 * 1 = disable
 */
void setDither()
{
  if (Dither == 0)
    {
      ChangeBit(dac, 0x0D, 7, 0);
      SerialUSB.println(F("Dithering enabled"));
    }
  else if (Dither == 1)
    {
      ChangeBit(dac, 0x0D, 7, 1);
      SerialUSB.println(F("Dithering disabled"));
    }
}

// -----------------------------  Function to enable or disable Jitter Eliminator / DPLL
/* 0 = disable
 * 1 = enable
 */
void setJitterElim()
{
  if (JitterElim == 0)
    {
      ChangeBit(dac, 0x0D, 5, 0);
      SerialUSB.println(F("Jitter Eliminator / DPLL disabled"));
    }
  else if (JitterElim == 1)
    {
      ChangeBit(dac, 0x0D, 5, 1);
      SerialUSB.println(F("Jitter Eliminator / DPLL enabled"));
    }
}

// -----------------------------  Function to enable or disable OSF
/* 0 = enable
 * 1 = disable
 */
void setOSF()
{
  if (bypass_osf == 0)
    {
      ChangeBit(dac, 0x25, 7, 0);
      SerialUSB.println(F("OSF enabled"));
    }
  else if (bypass_osf == 1)
    {
      ChangeBit(dac, 0x25, 7, 1);
      SerialUSB.println(F("OSF disabled"));
    }
}

// -----------------------------  Function to enable or disable automatic de-emphasis & set frequency
/* deemph_bypass:
 * 0 = enable deemphasis
 * 1 = disable deemphasis
 *  
 * auto_deemph:
 * 1 = enable
 * 0 = disable
 * 
 * deemph_sel:
 * 0 = 32KHz
 * 1 = 44.1KHz
 * 2 = 48KHz
 */
void setDeemphasis()
{
  if (deemph_bypass == 0)
    {
      ChangeBit(dac, 0x06, 6, 0);
      SerialUSB.println(F("De-emphasis filters enabled"));
    }
  else if (deemph_bypass == 1)
    {
      ChangeBit(dac, 0x06, 6, 1);
      SerialUSB.println(F("De-emphasis filters disabled"));
    }
  
  if (auto_deemph == 0)
    {
      ChangeBit(dac, 0x06, 7, 0);
      SerialUSB.println(F("Automatic de-emphasis disabled"));
    }
  else if (auto_deemph == 1)
    {
      ChangeBit(dac, 0x06, 7, 1);
      SerialUSB.println(F("Automatic de-emphasis enabled"));
    }

  if (deemph_sel == 0)
    {
      ChangeBit(dac, 0x06, 5, 0);
      ChangeBit(dac, 0x06, 4, 0);
      SerialUSB.println(F("Manual 32KHz de-emphasis enabled"));
    }
  else if (deemph_sel == 1)
    {
      ChangeBit(dac, 0x06, 5, 0);
      ChangeBit(dac, 0x06, 4, 1);
      SerialUSB.println(F("Manual 44.1KHz de-emphasis enabled"));
    }  
  else if (deemph_sel == 2)
    {
      ChangeBit(dac, 0x06, 5, 1);
      ChangeBit(dac, 0x06, 4, 0);
      SerialUSB.println(F("Manual 48KHz de-emphasis enabled"));
    }  
}

// -----------------------------  Function to determine Sampling Rate
/* sig_type:
 * 0 = DOP
 * 1 = SPDIF
 * 2 = I2S
 * 3 = DSD
*/
void getSR()
{
  volatile unsigned long DPLLNum; // Variable to hold DPLL value
  volatile unsigned long RegVal;  // Variable to hold Register values

  DPLLNum=0;
  // Reading the 4 registers of DPLL one byte at a time and stuffing into a single 32-bit number
  DPLLNum|=ReadRegister(dac, 0x45);
  DPLLNum<<=8;
  DPLLNum|=ReadRegister(dac, 0x44);
  DPLLNum<<=8;
  DPLLNum|=ReadRegister(dac, 0x43);
  DPLLNum<<=8;
  DPLLNum|=ReadRegister(dac, 0x42);

  SerialUSB.print(F("Raw DPLL number: ")); SerialUSB.println(DPLLNum);

  /*
  if (sig_type == 1)
  {
    DPLLNum*=20;      // Calculate SR for SPDIF 
    DPLLNum/=859;     // Calculate SR for SDPIF
  }
  else 
  {                  // Different calculation for SPDIF and I2S/DSD
    DPLLNum*=4;      // Calculate SR for I2S/DSD
    DPLLNum/=10995;  // Calculate SR for I2S/DSD
  }
  */

  long sampleRate = 0;
  sampleRate=(DPLLNum*MCLK)/42940;
    
  SerialUSB.print(F("Sample Rate: ")); SerialUSB.println(sampleRate);
  
  if (DPLLNum < 90000) // Adjusting because in integer operation, the residual is truncated
    DPLLNum+=1;
  else
    if (DPLLNum < 190000)
      DPLLNum+=2;
    else
      if (DPLLNum < 350000)
        DPLLNum+=3;
      else
        DPLLNum+=4;

  
  if ((sig_type==0) || (sig_type==3))             // SR calculation for DSD signals (either native or DoP)
    {
      if(sampleRate>461520)
          {
            SR = 0;
            if (input == 0)
              {
                DPLL_DSD = 15;
                setDPLLDSD();
              }
            else if (input == 1)
              {
                DPLL_DSD = 15;
                setDPLLDSD();
              }
            else if (input == 2)
              {
                DPLL_DSD = 15;
                setDPLLDSD();
              }                          
            else if (input == 3)
              {
                DPLL_DSD = 15;
                setDPLLDSD();
              }
            else if (input == 4)
              {
                DPLL_DSD = 15;
                setDPLLDSD();
              }               
            else if (input == 5)
              {
                DPLL_DSD = 15;
                setDPLLDSD();
              }               
            SerialUSB.println(F("Invalid DSD, DPLL set to 15"));
          }
        else
          if(sampleRate>451000)
                  {
                    SR = 1;
                    if (input == 0)
                      {
                        DPLL_DSD = Input[0].DPLLDSD1;
                        setDPLLDSD();
                      }
                    else if (input == 1)
                      {
                        DPLL_DSD = Input[1].DPLLDSD1;
                        setDPLLDSD();
                      }
                    else if (input == 2)
                      {
                        DPLL_DSD = Input[2].DPLLDSD1;
                        setDPLLDSD();
                      }
                    else if (input == 3)
                      {
                        DPLL_DSD = Input[3].DPLLDSD1;
                        setDPLLDSD();
                      }
                    else if (input == 4)
                      {
                        DPLL_DSD = Input[4].DPLLDSD1;
                        setDPLLDSD();
                      }
                    else if (input == 5)
                      {
                        DPLL_DSD = Input[5].DPLLDSD1;
                        setDPLLDSD();
                      }                                            
                    SerialUSB.println(F("44.8 MHz DSD"));
                  }
        else
          if(sampleRate>225700)
                  {
                    SR = 2;
                    if (input == 0)
                      {
                        DPLL_DSD = Input[0].DPLLDSD2;
                        setDPLLDSD();
                      }
                    else if (input == 1)
                      {
                        DPLL_DSD = Input[1].DPLLDSD2;
                        setDPLLDSD();
                      }
                    else if (input == 2)
                      {
                        DPLL_DSD = Input[2].DPLLDSD2;
                        setDPLLDSD();
                      }
                    else if (input == 3)
                      {
                        DPLL_DSD = Input[3].DPLLDSD2;
                        setDPLLDSD();
                      }
                    else if (input == 4)
                      {
                        DPLL_DSD = Input[4].DPLLDSD2;
                        setDPLLDSD();
                      }
                    else if (input == 5)
                      {
                        DPLL_DSD = Input[5].DPLLDSD2;
                        setDPLLDSD();
                      }                        
                    SerialUSB.println(F("22.4 MHz DSD"));
                  }
          else
            if(sampleRate>112000)
                  {
                    SR = 3;
                    if (input == 0)
                      {
                        DPLL_DSD = Input[0].DPLLDSD3;
                        setDPLLDSD();
                      }
                    else if (input == 1)
                      {
                        DPLL_DSD = Input[1].DPLLDSD3;
                        setDPLLDSD();
                      }
                    else if (input == 2)
                      {
                        DPLL_DSD = Input[2].DPLLDSD3;
                        setDPLLDSD();
                      }
                    else if (input == 3)
                      {
                        DPLL_DSD = Input[3].DPLLDSD3;
                        setDPLLDSD();
                      }
                    else if (input == 4)
                      {
                        DPLL_DSD = Input[4].DPLLDSD3;
                        setDPLLDSD();
                      }
                    else if (input == 5)
                      {
                        DPLL_DSD = Input[5].DPLLDSD3;
                        setDPLLDSD();
                      }                                            
                    SerialUSB.println(F("11.2 MHz DSD"));
                  }
            else
              if(sampleRate>56000)
                  {
                    SR = 4;
                    if (input == 0)
                      {
                        DPLL_DSD = Input[0].DPLLDSD4;
                        setDPLLDSD();
                      }
                    else if (input == 1)
                      {
                        DPLL_DSD = Input[1].DPLLDSD4;
                        setDPLLDSD();
                      }
                    else if (input == 2)
                      {
                        DPLL_DSD = Input[2].DPLLDSD4;
                        setDPLLDSD();
                      }
                    else if (input == 3)
                      {
                        DPLL_DSD = Input[3].DPLLDSD4;
                        setDPLLDSD();
                      }
                    else if (input == 4)
                      {
                        DPLL_DSD = Input[4].DPLLDSD4;
                        setDPLLDSD();
                      }
                    else if (input == 5)
                      {
                        DPLL_DSD = Input[5].DPLLDSD4;
                        setDPLLDSD();
                      }                        
                    SerialUSB.println(F("5.6 MHz DSD"));
                  }
              else
                if ((sampleRate>28000) || (sampleRate>1700))
                  {
                    SR = 5;
                    if (input == 0)
                      {
                        DPLL_DSD = Input[0].DPLLDSD5;
                        setDPLLDSD();
                      }
                    else if (input == 1)
                      {
                        DPLL_DSD = Input[1].DPLLDSD5;
                        setDPLLDSD();
                      }
                    else if (input == 2)
                      {
                        DPLL_DSD = Input[2].DPLLDSD5;
                        setDPLLDSD();
                      }
                    else if (input == 3)
                      {
                        DPLL_DSD = Input[3].DPLLDSD5;
                        setDPLLDSD();
                      }
                    else if (input == 4)
                      {
                        DPLL_DSD = Input[4].DPLLDSD5;
                        setDPLLDSD();
                      }
                    else if (input == 5)
                      {
                        DPLL_DSD = Input[5].DPLLDSD5;
                        setDPLLDSD();
                      }                        
                    SerialUSB.println(F("2.8 MHz DSD"));
                  }
                else
                  {
                    SR = 0;
                    if (input == 0)
                      {
                        DPLL_DSD = 15;
                        setDPLLDSD();
                      }
                    else if (input == 1)
                      {
                        DPLL_DSD = 15;
                        setDPLLDSD();
                      }
                    else if (input == 2)
                      {
                        DPLL_DSD = 15;
                        setDPLLDSD();
                      }
                    else if (input == 3)
                      {
                        DPLL_DSD = 15;
                        setDPLLDSD();
                      }
                    else if (input == 4)
                      {
                        DPLL_DSD = 15;
                        setDPLLDSD();
                      }
                    else if (input == 5)
                      {
                        DPLL_DSD = 15;
                        setDPLLDSD();
                      }                        
                    SerialUSB.println(F("Invalid DSD, DPLL set to 15."));
                  }
        }
    
  if (sig_type==1 || sig_type==2)         // If signal is PCM (either SPDIF or I2S)
      {
        if(sampleRate>7690)
          {
            SR = 0;
            if (input == 0)
              {
                DPLL_PCM = 15;
                setDPLLPCM();
              }
            else if (input == 1)
              {
                DPLL_PCM = 15;
                setDPLLPCM();
              }
            else if (input == 2)
              {
                DPLL_PCM = 15;
                setDPLLPCM();
              }
            else if (input == 3)
              {
                DPLL_PCM = 15;
                setDPLLPCM();
              }
            else if (input == 4)
              {
                DPLL_PCM = 15;
                setDPLLPCM();
              }
            else if (input == 5)
              {
                DPLL_PCM = 15;
                setDPLLPCM();
              }
            SerialUSB.println(F("Invalid SR, DPLL set to 15."));
          }
          else
            if(sampleRate>7675)
              {
                SR = 6;
                if (input == 0)
                  {
                    DPLL_PCM = Input[0].DPLLPCM1;
                    setDPLLPCM();
                  }
                else if (input == 1)
                  {
                    DPLL_PCM = Input[1].DPLLPCM1;
                    setDPLLPCM();
                  }
                else if (input == 2)
                  {
                    DPLL_PCM = Input[2].DPLLPCM1;
                    setDPLLPCM();
                  }
                else if (input == 3)
                  {
                    DPLL_PCM = Input[3].DPLLPCM1;
                    setDPLLPCM();
                  }
                else if (input == 4)
                  {
                    DPLL_PCM = Input[4].DPLLPCM1;
                    setDPLLPCM();
                  }
                else if (input == 5)
                  {
                    DPLL_PCM = Input[5].DPLLPCM1;
                    setDPLLPCM();
                  }                  
                SerialUSB.println(F("768K PCM"));
              }
            else
              if(sampleRate>7050)
                {
                  SR = 7;
                  if (input == 0)
                    {
                      DPLL_PCM = Input[0].DPLLPCM2;
                      setDPLLPCM();
                    }
                  else if (input == 1)
                    {
                      DPLL_PCM = Input[1].DPLLPCM2;
                      setDPLLPCM();
                    }
                  else if (input == 2)
                    {
                      DPLL_PCM = Input[2].DPLLPCM2;
                      setDPLLPCM();
                    }
                  else if (input == 3)
                    {
                      DPLL_PCM = Input[3].DPLLPCM2;
                      setDPLLPCM();
                    }
                  else if (input == 4)
                    {
                      DPLL_PCM = Input[4].DPLLPCM2;
                      setDPLLPCM();
                    }
                  else if (input == 5)
                    {
                      DPLL_PCM = Input[5].DPLLPCM2;
                      setDPLLPCM();
                    }                       
                  SerialUSB.println(F("705.6K PCM"));
              }                  
            else
              if(sampleRate>3835)
                {
                  SR = 8;
                  if (input == 0)
                    {
                      DPLL_PCM = Input[0].DPLLPCM3;
                      setDPLLPCM();
                    }
                  else if (input == 1)
                    {
                      DPLL_PCM = Input[1].DPLLPCM3;
                      setDPLLPCM();
                    }
                  else if (input == 2)
                    {
                      DPLL_PCM = Input[2].DPLLPCM3;
                      setDPLLPCM();
                    }
                  else if (input == 3)
                    {
                      DPLL_PCM = Input[3].DPLLPCM3;
                      setDPLLPCM();
                    }
                  else if (input == 4)
                    {
                      DPLL_PCM = Input[4].DPLLPCM3;
                      setDPLLPCM();
                    }
                  else if (input == 5)
                    {
                      DPLL_PCM = Input[5].DPLLPCM3;
                      setDPLLPCM();
                    }                                        
                    SerialUSB.println(F("384K PCM"));
                }       
            else
              if(sampleRate>3510)
                {
                  SR = 9;
                  if (input == 0)
                    {
                      DPLL_PCM = Input[0].DPLLPCM4;
                      setDPLLPCM();
                    }
                  else if (input == 1)
                    {
                      DPLL_PCM = Input[1].DPLLPCM4;
                      setDPLLPCM();
                    }
                  else if (input == 2)
                    {
                      DPLL_PCM = Input[2].DPLLPCM4;
                      setDPLLPCM();
                    }
                  else if (input == 3)
                    {
                      DPLL_PCM = Input[3].DPLLPCM4;
                      setDPLLPCM();
                    }
                  else if (input == 4)
                    {
                      DPLL_PCM = Input[4].DPLLPCM4;
                      setDPLLPCM();
                    }
                  else if (input == 5)
                    {
                      DPLL_PCM = Input[5].DPLLPCM4;
                      setDPLLPCM();
                    }                                   
                  SerialUSB.println(F("352.8K PCM"));
                }
            else
              if(sampleRate>1910)
                {
                  SR = 10;
                  if (input == 0)
                    {
                      DPLL_PCM = Input[0].DPLLPCM5;
                      setDPLLPCM();
                    }
                  else if (input == 1)
                    {
                      DPLL_PCM = Input[1].DPLLPCM5;
                      setDPLLPCM();
                    }
                  else if (input == 2)
                    {
                      DPLL_PCM = Input[2].DPLLPCM5;
                      setDPLLPCM();
                    }
                  else if (input == 3)
                    {
                      DPLL_PCM = Input[3].DPLLPCM5;
                      setDPLLPCM();
                    }
                  else if (input == 4)
                    {
                      DPLL_PCM = Input[4].DPLLPCM5;
                      setDPLLPCM();
                    }
                  else if (input == 5)
                    {
                      DPLL_PCM = Input[5].DPLLPCM5;
                      setDPLLPCM();
                    }                               
                  SerialUSB.println(F("192K PCM"));
                }
            else
              if(sampleRate>1756)
                {
                  SR = 11;
                  if (input == 0)
                    {
                      DPLL_PCM = Input[0].DPLLPCM6;
                      setDPLLPCM();
                    }
                  else if (input == 1)
                    {
                      DPLL_PCM = Input[1].DPLLPCM6;
                      setDPLLPCM();
                    }
                  else if (input == 2)
                    {
                      DPLL_PCM = Input[2].DPLLPCM6;
                      setDPLLPCM();
                    }
                  else if (input == 3)
                    {
                      DPLL_PCM = Input[3].DPLLPCM6;
                      setDPLLPCM();
                    }
                  else if (input == 4)
                    {
                      DPLL_PCM = Input[4].DPLLPCM6;
                      setDPLLPCM();
                    }
                  else if (input == 5)
                    {
                      DPLL_PCM = Input[5].DPLLPCM6;
                      setDPLLPCM();
                    }                   
                  SerialUSB.println(F("176.4K PCM"));
                }                        
            else
              if(sampleRate>954)
                {
                  SR = 12;
                  if (input == 0)
                    {
                      DPLL_PCM = Input[0].DPLLPCM7;
                      setDPLLPCM();
                    }
                  else if (input == 1)
                    {
                      DPLL_PCM = Input[1].DPLLPCM7;
                      setDPLLPCM();
                    }
                  else if (input == 2)
                    {
                      DPLL_PCM = Input[2].DPLLPCM7;
                      setDPLLPCM();
                    }
                  else if (input == 3)
                    {
                      DPLL_PCM = Input[3].DPLLPCM7;
                      setDPLLPCM();
                    }
                  else if (input == 4)
                    {
                      DPLL_PCM = Input[4].DPLLPCM7;
                      setDPLLPCM();
                    }
                  else if (input == 5)
                    {
                      DPLL_PCM = Input[5].DPLLPCM7;
                      setDPLLPCM();
                    }                   
                  SerialUSB.println(F("96K PCM"));
                } 
            else
              if(sampleRate>878)
                {
                  SR = 13;
                  if (input == 0)
                    {
                      DPLL_PCM = Input[0].DPLLPCM8;
                      setDPLLPCM();
                    }
                  else if (input == 1)
                    {
                      DPLL_PCM = Input[1].DPLLPCM8;
                      setDPLLPCM();
                    }
                  else if (input == 2)
                    {
                      DPLL_PCM = Input[2].DPLLPCM8;
                      setDPLLPCM();
                    }                   
                  else if (input == 3)
                    {
                      DPLL_PCM = Input[3].DPLLPCM8;
                      setDPLLPCM();
                    }
                  else if (input == 4)
                    {
                      DPLL_PCM = Input[4].DPLLPCM8;
                      setDPLLPCM();
                    }
                  else if (input == 5)
                    {
                      DPLL_PCM = Input[5].DPLLPCM8;
                      setDPLLPCM();
                    }  
                  SerialUSB.println(F("88.2K PCM"));
                } 
            else
              if(sampleRate>475)
                {
                  SR = 14;
                  if (input == 0)
                    {
                      DPLL_PCM = Input[0].DPLLPCM9;
                      setDPLLPCM();
                    }
                  else if (input == 1)
                    {
                      DPLL_PCM = Input[1].DPLLPCM9;
                      setDPLLPCM();
                    }
                  else if (input == 2)
                    {
                      DPLL_PCM = Input[2].DPLLPCM9;
                      setDPLLPCM();
                    }                   
                  else if (input == 3)
                    {
                      DPLL_PCM = Input[3].DPLLPCM9;
                      setDPLLPCM();
                    }
                  else if (input == 4)
                    {
                      DPLL_PCM = Input[4].DPLLPCM9;
                      setDPLLPCM();
                    }
                  else if (input == 5)
                    {
                      DPLL_PCM = Input[5].DPLLPCM9;
                      setDPLLPCM();
                    }  
                  SerialUSB.println(F("48K PCM"));
                } 
            else
              if(sampleRate>430)
                {
                  SR = 15;
                  if (input == 0)
                    {
                      DPLL_PCM = Input[0].DPLLPCM10;
                      setDPLLPCM();
                    }
                  else if (input == 1)
                    {
                      DPLL_PCM = Input[1].DPLLPCM10;
                      setDPLLPCM();
                    }
                  else if (input == 2)
                    {
                      DPLL_PCM = Input[2].DPLLPCM10;
                      setDPLLPCM();
                    }                   
                  else if (input == 3)
                    {
                      DPLL_PCM = Input[3].DPLLPCM10;
                      setDPLLPCM();
                    }
                  else if (input == 4)
                    {
                      DPLL_PCM = Input[4].DPLLPCM10;
                      setDPLLPCM();
                    }
                  else if (input == 5)
                    {
                      DPLL_PCM = Input[5].DPLLPCM10;
                      setDPLLPCM();
                    }  
                  SerialUSB.println(F("44.1K PCM"));
                } 
            else
              if(sampleRate>310)
                {
                  SR = 16;
                  if (input == 0)
                    {
                      DPLL_PCM = Input[0].DPLLPCM11;
                      setDPLLPCM();
                    }
                  else if (input == 1)
                    {
                      DPLL_PCM = Input[1].DPLLPCM11;
                      setDPLLPCM();
                    }
                  else if (input == 2)
                    {
                      DPLL_PCM = Input[2].DPLLPCM11;
                      setDPLLPCM();
                    }                   
                  else if (input == 3)
                    {
                      DPLL_PCM = Input[3].DPLLPCM11;
                      setDPLLPCM();
                    }
                  else if (input == 4)
                    {
                      DPLL_PCM = Input[4].DPLLPCM11;
                      setDPLLPCM();
                    }
                  else if (input == 5)
                    {
                      DPLL_PCM = Input[5].DPLLPCM11;
                      setDPLLPCM();
                    }  
                  SerialUSB.println(F("32K PCM"));
                } 
  }
}

// -----------------------------  Function to check lock status and determine signal type
/* Lock:
 * 0 = unlocked 
 * 1 = locked
 * 
 * sig_type:
 * 0 = DOP
 * 1 = SPDIF
 * 2 = I2S
 * 3 = DSD
 */
void getLockStatus()
{
  byte r = ReadRegister(dac, 0x40);       // Read Reg64
  if (bitRead(r,0) == 1)       // DAC has lock
    {
      lock = 1;
      r = ReadRegister(dac, 0x64);
      //SerialUSB.print("Reg100: "); SerialUSB.println(r, BIN);
      //r = r << 4;
      if (bitRead(r, 3) == 1)
        {
          sig_type = 0;                 // Decoded signal type is DoP      
        }
      else if (bitRead(r, 2) == 1)
        {
          sig_type = 1;                 // Decoded signal type is SPDIF      
        }
      else if (bitRead(r, 1) == 1)
        {
          sig_type = 2;                 // Decoded signal type is I2S
        }      
      else if (bitRead(r, 0) == 1)
        {
          sig_type = 3;                 // Decoded signal type is DSD
        }
        SerialUSB.print(F("DAC has locked to a signal of type "));
        SerialUSB.println(sig_type);
    }
    else if (bitRead(r,0) == 0)// DAC has no lock
      {
        SerialUSB.println(F("No Lock"));
        lock = 0;
      }
}

// ----------------------------- Function that sets the screen's backlight level --------------------------------------------------------------------
void setBacklight()
  {
    if (backlight==10) analogWrite(DIMMPIN, 0);            // Set LCD backlight to full.
    else if (backlight==9) analogWrite(DIMMPIN, 22);       // Set LCD backlight to 90%.
    else if (backlight==8) analogWrite(DIMMPIN, 45);       // Set LCD backlight to 80%.
    else if (backlight==7) analogWrite(DIMMPIN, 70);       // Set LCD backlight to 70%.
    else if (backlight==6) analogWrite(DIMMPIN, 100);      // Set LCD backlight to 60%.
    else if (backlight==5) analogWrite(DIMMPIN, 128);      // Set LCD backlight to 50%.
    else if (backlight==4) analogWrite(DIMMPIN, 160);      // Set LCD backlight to 40%.
    else if (backlight==3) analogWrite(DIMMPIN, 190);      // Set LCD backlight to 30%.
    else if (backlight==2) analogWrite(DIMMPIN, 210);      // Set LCD backlight to 20%.
    else if (backlight==1) analogWrite(DIMMPIN, 230);      // Set LCD backlight to 10%.
    else if (backlight==0) analogWrite(DIMMPIN, 255);      // Set LCD backlight to off.
  }

// ----------------------------- Function that reads a character from the serial port ------------------------------------------------------------
char getCommand()
{
  char c = '\0';
  if (SerialUSB.available())
  {
    c = SerialUSB.read();
  }
  return c;
}

// ----------------------------- Function that displays available commands through the serial port -----------------------------------------------
void displayHelp()
{
  if (dac_chip == 0)
    {
      SerialUSB.print("ES9038Pro");
    }
  else if (dac_chip == 1)
    {
      SerialUSB.print("ES9028Pro");
    }
  SerialUSB.println(F(" DAC Controller"));
  SerialUSB.println();
  SerialUSB.println(F("Press a to select Input 1"));
  SerialUSB.println(F("Press b to select Input 2"));
  SerialUSB.println(F("Press c to select Input 3"));
  SerialUSB.println(F("Press d to set IIR bandwidth to 47.44KHz"));
  SerialUSB.println(F("Press e to set IIR bandwidth to 50KHz"));
  SerialUSB.println(F("Press f to set IIR bandwidth to 60KHz"));
  SerialUSB.println(F("Press g to set IIR bandwidth to 70KHz"));
  SerialUSB.println(F("Press h to unmute the DAC"));
  SerialUSB.println(F("Press i to mute the DAC"));
  SerialUSB.println(F("Press j to get Lock status and calculate the SR"));
  SerialUSB.println(F("Press 0 to 6 to select filter"));
  SerialUSB.println(F("Press y to display all registers"));
  SerialUSB.println(F("Press z to display this menu"));
  SerialUSB.println(F("Press ! to restore EEPROM values to defaults"));
  SerialUSB.println(F("Press @ to initialize the DAC"));
  SerialUSB.println(F("Press ^ to test read EEPROM contents"));
  SerialUSB.println();
}

// ----------------------------- Function that resets and initializes the DAC upon powerup ---------------------------------------------------------------
void DeviceInit()
{  
  myGLCD.clrScr();
  myGLCD.setFont(calibri_bold_40);
  myGLCD.setColor(0, 50, 255);
  myGLCD.setBackColor(0, 0, 0);
  myGLCD.print("T F T", (tft_x/2)-(myGLCD.getStringWidth("T F T")/2), 10);
  myGLCD.print("H i F i D U I N O  P R O", (tft_x/2)-(myGLCD.getStringWidth("H i F i D U I N O  P R O")/2), 55);
  myGLCD.print("ES9028 & ES9038", (tft_x/2)-(myGLCD.getStringWidth("ES9028 & ES9038")/2), 110);
  #ifdef BIG
    myGLCD.print("v.1.08", (tft_x/2)-(myGLCD.getStringWidth("v.1.06")/2), 220);
  #else
    myGLCD.print("v.1.08", (tft_x/2)-(myGLCD.getStringWidth("v.1.06")/2), 200);
  #endif BIG

  Wire.beginTransmission(dac);
  error = Wire.endTransmission();
  myGLCD.setColor(0, 150, 255);
  if (error == 0)
  {
    byte r = ReadRegister(dac, 0x40);
    r = r >> 2;
    if (r == B101010)
      {
        SerialUSB.println(F("Found DAC ES9038Pro"));
        myGLCD.print("Found ES9038Pro", (tft_x/2)-(myGLCD.getStringWidth("Found ES9038Pro")/2), 160);
        dac_chip = 0;
      }
    else if ((r == B101001) || (r == B101000))
      {
        SerialUSB.println(F("Found DAC ES9028Pro"));
        myGLCD.print("Found ES9028Pro", (tft_x/2)-(myGLCD.getStringWidth("Found ES9028Pro")/2), 160);
        dac_chip = 1;
      }
  }
  else if (error==4) 
  {
    SerialUSB.println(F("Unknown error at DAC address."));
    myGLCD.print("Unknown error at DAC address.", (tft_x/2)-(myGLCD.getStringWidth("Unknown error at DAC address.")/2), 160);
  }
  else
  {
    SerialUSB.println(F("No response from DAC."));
    myGLCD.print("No response from DAC", (tft_x/2)-(myGLCD.getStringWidth("No response from DAC")/2), 150);
  }

  delay(1500);

  // Load test data to determine if it's the first time the code runs
  eeAddress = eeAddressT;
  byte *pinit = (byte*)&EpromTest;
  myEEPROM.read(eeAddress, pinit, sizeof(InitTest));

  SerialUSB.println(F("EEPROM test data:"));
  SerialUSB.println(EpromTest.test1);
  SerialUSB.println(EpromTest.test2);
  SerialUSB.println(EpromTest.test3);
  
  if ((EpromTest.test1 != 1) && (EpromTest.test2 != 2) && (EpromTest.test3 != 3))
    {
      SerialUSB.println(F("First run detected, initializing EEPROM.."));
      initEEPROM();
      delay(200);
      initEEPROM();         // For good measure..
    }
  
  // Load Input_1 settings from EEPROM
  eeAddress = eeAddress1;
  byte *pData = (byte*)&Input[0];
  myEEPROM.read(eeAddress, pData, sizeof(DAC_Input));
  SerialUSB.println("Test read Input_1 from EEPROM: ");
  SerialUSB.println(Input[0].filter_val);
  
  // Load Input_2 settings from EEPROM
  eeAddress = eeAddress2;
  byte *pData2 = (byte*)&Input[1];
  myEEPROM.read(eeAddress, pData2, sizeof(DAC_Input));
  SerialUSB.println("Test read Input_2 from EEPROM: ");
  SerialUSB.println(Input[1].filter_val);
  
  // Load Input_3 settings from EEPROM
  eeAddress = eeAddress3;
  byte *pData3 = (byte*)&Input[2];
  myEEPROM.read(eeAddress, pData3, sizeof(DAC_Input));

  // Load Input_4 settings from EEPROM
  eeAddress = eeAddress4;
  byte *pData4 = (byte*)&Input[3];
  myEEPROM.read(eeAddress, pData4, sizeof(DAC_Input));

  // Load Input_5 settings from EEPROM
  eeAddress = eeAddress5;
  byte *pData5 = (byte*)&Input[4];
  myEEPROM.read(eeAddress, pData5, sizeof(DAC_Input));

  // Load Input_6 settings from EEPROM
  eeAddress = eeAddress6;
  byte *pData6 = (byte*)&Input[5];
  myEEPROM.read(eeAddress, pData6, sizeof(DAC_Input));

// Load Settings1 from EEPROM
  eeAddress = eeAddressS;
  byte *pSettings = (byte*)&Settings1;
  myEEPROM.read(eeAddress, pSettings, sizeof(DAC_Settings));

  testEEPROM();
  
  SerialUSB.println(F("Test read Input_1 from EEPROM: "));
  SerialUSB.println(Input[0].filter_val);
  SerialUSB.println(Input[0].lockspeed_val);
  SerialUSB.println(F("Test read Input_6 from EEPROM: "));
  SerialUSB.println(Input[5].filter_val);
  SerialUSB.println(Input[5].lockspeed_val);

  SerialUSB.print(F("Initializing "));
  if (dac_chip == 0)
    {
      SerialUSB.println(F("ES9038Pro"));
    }
  else if (dac_chip == 1)
    {
      SerialUSB.println(F("ES9028Pro"));
    }
  
  ChangeBit(dac, 0x00, 0, 1);                   // Put DAC in RESET mode
  delay(500);                                   // Wait for 500ms
  ChangeBit(dac, 0x00, 0, 0);                   // Take DAC out of RESET mode
  mute = 1;                                     // Put DAC in mute
  setMute();                                    // Put DAC in mute
  ChangeBit(dac, 0x09, 7, 0);                   // Setup GPIO4
  ChangeBit(dac, 0x09, 6, 1);                   // Setup GPIO4
  ChangeBit(dac, 0x09, 5, 0);                   // Setup GPIO4
  ChangeBit(dac, 0x09, 4, 0);                   // Setup GPIO4  
  ChangeBit(dac, 0x0F, 2, 1);                   // Set DAC up for Stereo Mode
  ChangeBit(dac, 0x36, 7, 0);                   // Enable DoP support
  mute = 0;                                     // Take DAC out of mute
  setMute();                                    // Take DAC out of mute
  backlight = Settings1.backlight;
  setBacklight();
  preamp = Settings1.variable;
  preamp_def_vol = Settings1.preamp_def_vol;

  if (preamp == true)
    {
      volume = preamp_def_vol;
    }

  input = 0;
  setInput();
  setVol();
  main_disp(true);
}

// ----------------------------- Function that reads a register's data value --------------------------------------------------------------------
byte ReadRegister(int devaddr, byte regaddr)
  {                              
    Wire.beginTransmission(devaddr);
    Wire.write(regaddr);
    Wire.endTransmission();
    Wire.requestFrom(devaddr, 1);                 // only one byte
    if(Wire.available())                          // Wire.available indicates if data is available
      return Wire.read();                         // Wire.read() reads the data on the wire
    else
    return 9999;                                  // If no data in the wire, then return 9999 to indicate error
  }

// ----------------------------- Function that writes a data value to a register --------------------------------------------------------------------
void WriteRegister(int devaddr, byte regaddr, byte dataval)
  {
    Wire.beginTransmission(devaddr); // device
    Wire.write(regaddr); // register
    Wire.write(dataval); // data
    Wire.endTransmission();
  }

// ----------------------------- Function that changes a single bit in a register --------------------------------------------------------------------
void ChangeBit(int devaddr, byte regaddr, int data, boolean setting)
  {
    byte r = ReadRegister(devaddr, regaddr);
    if (setting == 1)
      {
        bitSet(r, data);
      } else
      if (setting == 0)
        {
          bitClear(r, data);
        }
      WriteRegister(devaddr, regaddr, r);
  }

// ----------------------------- Function that prints the values of registers for debugging --------------------------------------------------------------------
void PrintDebugReg(byte dac)
{
SerialUSB.println("");
SerialUSB.print(F("Register values for DAC: "));
SerialUSB.println(dac, HEX);

byte r = 0;

SerialUSB.print(F("Register 07: "));
r = ReadRegister(dac, 0x07);
SerialUSB.println(r,BIN);
r = 0;

SerialUSB.print(F("Register 12: "));
r = ReadRegister(dac, 0x0C);
SerialUSB.println(r,BIN);
r = 0;

SerialUSB.print(F("Register 13: "));
r = ReadRegister(dac, 0x0D);
SerialUSB.println(r,BIN);
r = 0;

SerialUSB.print(F("Register 37: "));
r = ReadRegister(dac, 0x25);
SerialUSB.println(r,BIN);
r = 0;

SerialUSB.print(F("Register 37 (dec): "));
r = ReadRegister(dac, 37);
SerialUSB.println(r,BIN);
r = 0;

SerialUSB.print(F("Register 64: "));
r = ReadRegister(dac, 0x40);
SerialUSB.println(r,BIN);
r = 0;

SerialUSB.print(F("Register 100: "));
r = ReadRegister(dac, 0x64);
SerialUSB.println(r,BIN);
r = 0;
}

// ----------------------------- Function that initalizes the EEPROM to default values for all inputs --------------------------------------------------------------------
void initEEPROM()
  {
    SerialUSB.print(F("Size of the DAC_Input structure: "));
    SerialUSB.println(sizeof(DAC_Input));
    
    // For Input 1
    eeAddress=eeAddress1;                  // Address of Input 1 settings
    Input[0].filter_val=0;
    Input[0].IIR_val=0;
    Input[0].DPLLPCM1=5; Input[0].DPLLPCM2=4; Input[0].DPLLPCM3=2; Input[0].DPLLPCM4=2; Input[0].DPLLPCM5=1; Input[0].DPLLPCM6=1; Input[0].DPLLPCM7=1; Input[0].DPLLPCM8=1; Input[0].DPLLPCM9=1; Input[0].DPLLPCM10=1; Input[0].DPLLPCM11=1;
    Input[0].DPLLDSD1=6; Input[0].DPLLDSD2=6; Input[0].DPLLDSD3=6; Input[0].DPLLDSD4=6; Input[0].DPLLDSD5=6;
    Input[0].lockspeed_val=0;
    Input[0].dither_val=0;
    Input[0].jitter_elim_val=1;
    Input[0].bypass_osf_val=0;
    Input[0].deemph_bypass_val=1;
    Input[0].auto_deemph_val=0;
    Input[0].deemph_sel_val=0;
    myEEPROM.write(eeAddress, (byte *)&Input[0], sizeof(DAC_Input));    // write to EEPROM
    SerialUSB.println(F("Input 1 initialized"));

    // For Input 2
    eeAddress=eeAddress2;                 // Address of Input 2 settings
    Input[1].filter_val=0;
    Input[1].IIR_val=0;
    Input[1].DPLLPCM1=5; Input[1].DPLLPCM2=4; Input[1].DPLLPCM3=2; Input[1].DPLLPCM4=2; Input[1].DPLLPCM5=1; Input[1].DPLLPCM6=1; Input[1].DPLLPCM7=1; Input[1].DPLLPCM8=1; Input[1].DPLLPCM9=1; Input[1].DPLLPCM10=1; Input[1].DPLLPCM11=1;
    Input[1].DPLLDSD1=6; Input[1].DPLLDSD2=6; Input[1].DPLLDSD3=6; Input[1].DPLLDSD4=6; Input[1].DPLLDSD5=6;
    Input[1].lockspeed_val=0;
    Input[1].dither_val=0;
    Input[1].jitter_elim_val=1;
    Input[1].bypass_osf_val=0;
    Input[1].deemph_bypass_val=1;
    Input[1].auto_deemph_val=0;
    Input[1].deemph_sel_val=0;
    myEEPROM.write(eeAddress, (byte *)&Input[1], sizeof(DAC_Input));    // write to EEPROM
    SerialUSB.println(F("Input 2 initialized"));
    
    // For Input 3
    eeAddress=eeAddress3;               // Address of Input 3 settings
    Input[2].filter_val=4;
    Input[2].IIR_val=0;
    Input[2].DPLLPCM1=3; Input[2].DPLLPCM2=3; Input[2].DPLLPCM3=3; Input[2].DPLLPCM4=3; Input[2].DPLLPCM5=3; Input[2].DPLLPCM6=3; Input[2].DPLLPCM7=3; Input[2].DPLLPCM8=3; Input[2].DPLLPCM9=3; Input[2].DPLLPCM10=3; Input[2].DPLLPCM11=3;
    Input[2].DPLLDSD1=6; Input[2].DPLLDSD2=6; Input[2].DPLLDSD3=6; Input[2].DPLLDSD4=6; Input[2].DPLLDSD5=6;
    Input[2].lockspeed_val=0;
    Input[2].dither_val=0;
    Input[2].jitter_elim_val=1;    
    Input[2].bypass_osf_val=0;
    Input[2].deemph_bypass_val=1;
    Input[2].auto_deemph_val=0;
    Input[2].deemph_sel_val=0;    
    myEEPROM.write(eeAddress, (byte *)&Input[2], sizeof(DAC_Input));    // write to EEPROM
    SerialUSB.println(F("Input 3 initialized"));

    // For Input 4
    eeAddress=eeAddress4;               // Address of Input 4 settings
    Input[3].filter_val=4;
    Input[3].IIR_val=0;
    Input[3].DPLLPCM1=3; Input[3].DPLLPCM2=3; Input[3].DPLLPCM3=3; Input[3].DPLLPCM4=3; Input[3].DPLLPCM5=3; Input[3].DPLLPCM6=3; Input[3].DPLLPCM7=3; Input[3].DPLLPCM8=3; Input[3].DPLLPCM9=3; Input[3].DPLLPCM10=3; Input[3].DPLLPCM11=3;
    Input[3].DPLLDSD1=6; Input[3].DPLLDSD2=6; Input[3].DPLLDSD3=6; Input[3].DPLLDSD4=6; Input[3].DPLLDSD5=6;
    Input[3].lockspeed_val=0;
    Input[3].dither_val=0;
    Input[3].jitter_elim_val=1;    
    Input[3].bypass_osf_val=0;
    Input[3].deemph_bypass_val=1;
    Input[3].auto_deemph_val=0;
    Input[3].deemph_sel_val=0;    
    myEEPROM.write(eeAddress, (byte *)&Input[3], sizeof(DAC_Input));    // write to EEPROM
    SerialUSB.println(F("Input 4 initialized"));
    
    // For Input 5
    eeAddress=eeAddress5;               // Address of Input 5 settings
    Input[4].filter_val=4;
    Input[4].IIR_val=0;
    Input[4].DPLLPCM1=3; Input[4].DPLLPCM2=3; Input[4].DPLLPCM3=3; Input[4].DPLLPCM4=3; Input[4].DPLLPCM5=3; Input[4].DPLLPCM6=3; Input[4].DPLLPCM7=3; Input[4].DPLLPCM8=3; Input[4].DPLLPCM9=3; Input[4].DPLLPCM10=3; Input[4].DPLLPCM11=3;
    Input[4].DPLLDSD1=6; Input[4].DPLLDSD2=6; Input[4].DPLLDSD3=6; Input[4].DPLLDSD4=6; Input[4].DPLLDSD5=6;
    Input[4].lockspeed_val=0;
    Input[4].dither_val=0;
    Input[4].jitter_elim_val=1;    
    Input[4].bypass_osf_val=0;
    Input[4].deemph_bypass_val=1;
    Input[4].auto_deemph_val=0;
    Input[4].deemph_sel_val=0;    
    myEEPROM.write(eeAddress, (byte *)&Input[4], sizeof(DAC_Input));    // write to EEPROM
    SerialUSB.println(F("Input 5 initialized"));
    
    // For Input 6
    eeAddress=eeAddress6;               // Address of Input 6 settings
    Input[5].filter_val=4;
    Input[5].IIR_val=0;
    Input[5].DPLLPCM1=3; Input[5].DPLLPCM2=3; Input[5].DPLLPCM3=3; Input[5].DPLLPCM4=3; Input[5].DPLLPCM5=3; Input[5].DPLLPCM6=3; Input[5].DPLLPCM7=3; Input[5].DPLLPCM8=3; Input[5].DPLLPCM9=3; Input[5].DPLLPCM10=3; Input[5].DPLLPCM11=3;
    Input[5].DPLLDSD1=6; Input[5].DPLLDSD2=6; Input[5].DPLLDSD3=6; Input[5].DPLLDSD4=6; Input[5].DPLLDSD5=6;
    Input[5].lockspeed_val=0;
    Input[5].dither_val=0;
    Input[5].jitter_elim_val=1;    
    Input[5].bypass_osf_val=0;
    Input[5].deemph_bypass_val=1;
    Input[5].auto_deemph_val=0;
    Input[5].deemph_sel_val=0;    
    myEEPROM.write(eeAddress, (byte *)&Input[5], sizeof(DAC_Input));    // write to EEPROM
    SerialUSB.println(F("Input 6 initialized"));
    
    // For Settings1
    eeAddress=eeAddressS;                // Address of Settings1
    Settings1.variable=1;         // Store default preamp mode setting to EEPROM
    Settings1.preamp_def_vol=20;  // Store default preamp volume setting to EEPROM
    Settings1.backlight=10;       // Store default TFT backlight setting to EEPROM
    myEEPROM.write(eeAddress, (byte *)&Settings1, sizeof(DAC_Settings));    // write to EEPROM
    SerialUSB.println(F("Settings1 initialized"));
    
    // To tell if EEPROM is initialized or not
    eeAddress=eeAddressT;                // Address of init test data
    EpromTest.test1=1;
    EpromTest.test2=2;
    EpromTest.test3=3;
    myEEPROM.write(eeAddress, (byte *)&EpromTest, sizeof(InitTest));    // write to EEPROM
    
  }

// ----------------------------- Function that saves the changed setting to EEPROM --------------------------------------------------------------------
void savetoEEPROM()
  {
   
    if ((input == 0) && ((select1Mode==true) || (do_select_filter == true) || (selectDPLLMode==true)))      // If saving Input 1
      {
        eeAddress=eeAddress1;                                                     // Address of Input 1 settings
        Input[0].filter_val=filter;
        Input[0].IIR_val=IIR;
        Input[0].lockspeed_val=LockSpeed;
        Input[0].dither_val=Dither;
        Input[0].jitter_elim_val=JitterElim;
        Input[0].bypass_osf_val=bypass_osf;
        Input[0].deemph_bypass_val=deemph_bypass;
        Input[0].auto_deemph_val=auto_deemph;
        Input[0].deemph_sel_val=deemph_sel;
        myEEPROM.write(eeAddress, (byte *)&Input[0], sizeof(DAC_Input));    // write to EEPROM
        SerialUSB.println(F("Input 1 updated"));
      }

    else if ((input == 1) && ((select1Mode==true) || (do_select_filter == true) || (selectDPLLMode==true))) // If saving Input 2
      {
        eeAddress=eeAddress2;                                                     // Address of Input 2 settings
        Input[1].filter_val=filter;
        Input[1].IIR_val=IIR;
        Input[1].lockspeed_val=LockSpeed;
        Input[1].dither_val=Dither;
        Input[1].jitter_elim_val=JitterElim;
        Input[1].bypass_osf_val=bypass_osf;
        Input[1].deemph_bypass_val=deemph_bypass;
        Input[1].auto_deemph_val=auto_deemph;
        Input[1].deemph_sel_val=deemph_sel;
        myEEPROM.write(eeAddress, (byte *)&Input[1], sizeof(DAC_Input));    // write to EEPROM
        SerialUSB.println(F("Input 2 updated"));
      }

    else if ((input == 2) && ((select1Mode==true) || (do_select_filter == true) || (selectDPLLMode==true))) // If saving Input 3
      {
        eeAddress=eeAddress3;                                                     // Address of Input 3 settings
        Input[2].filter_val=filter;
        Input[2].IIR_val=IIR;
        Input[2].lockspeed_val=LockSpeed;
        Input[2].dither_val=Dither;
        Input[2].jitter_elim_val=JitterElim;    
        Input[2].bypass_osf_val=bypass_osf;
        Input[2].deemph_bypass_val=deemph_bypass;
        Input[2].auto_deemph_val=auto_deemph;
        Input[2].deemph_sel_val=deemph_sel;    
        myEEPROM.write(eeAddress, (byte *)&Input[2], sizeof(DAC_Input));    // write to EEPROM
        SerialUSB.println(F("Input 3 updated"));
      }
    else if ((input == 3) && ((select1Mode==true) || (do_select_filter == true) || (selectDPLLMode==true))) // If saving Input 4
      {
        eeAddress=eeAddress4;                                                     // Address of Input 4 settings
        Input[3].filter_val=filter;
        Input[3].IIR_val=IIR;
        Input[3].lockspeed_val=LockSpeed;
        Input[3].dither_val=Dither;
        Input[3].jitter_elim_val=JitterElim;
        Input[3].bypass_osf_val=bypass_osf;
        Input[3].deemph_bypass_val=deemph_bypass;
        Input[3].auto_deemph_val=auto_deemph;
        Input[3].deemph_sel_val=deemph_sel;
        myEEPROM.write(eeAddress, (byte *)&Input[3], sizeof(DAC_Input));    // write to EEPROM
        SerialUSB.println(F("Input 4 updated"));
      }
    else if ((input == 4) && ((select1Mode==true) || (do_select_filter == true) || (selectDPLLMode==true))) // If saving Input 5
      {
        eeAddress=eeAddress5;                                                     // Address of Input 5 settings
        Input[4].filter_val=filter;
        Input[4].IIR_val=IIR;
        Input[4].lockspeed_val=LockSpeed;
        Input[4].dither_val=Dither;
        Input[4].jitter_elim_val=JitterElim;
        Input[4].bypass_osf_val=bypass_osf;
        Input[4].deemph_bypass_val=deemph_bypass;
        Input[4].auto_deemph_val=auto_deemph;
        Input[4].deemph_sel_val=deemph_sel;   
        myEEPROM.write(eeAddress, (byte *)&Input[4], sizeof(DAC_Input));    // write to EEPROM
        SerialUSB.println(F("Input 5 updated"));
      }

    else if ((input == 5) && ((select1Mode==true) || (do_select_filter == true) || (selectDPLLMode==true))) // If saving Input 6
      {
        eeAddress=eeAddress6;                                                     // Address of Input 6 settings
        Input[5].filter_val=filter;
        Input[5].IIR_val=IIR;
        Input[5].lockspeed_val=LockSpeed;
        Input[5].dither_val=Dither;
        Input[5].jitter_elim_val=JitterElim;
        Input[5].bypass_osf_val=bypass_osf;
        Input[5].deemph_bypass_val=deemph_bypass;
        Input[5].auto_deemph_val=auto_deemph;
        Input[5].deemph_sel_val=deemph_sel;  
        myEEPROM.write(eeAddress, (byte *)&Input[5], sizeof(DAC_Input));    // write to EEPROM
        SerialUSB.println(F("Input 6 updated"));
      }

    else if (select2Mode==true)               // If saving Settings
      {
        eeAddress=eeAddressS;                            // Address of Settings1
        //size = sizeof(Settings1);
        SerialUSB.print(F("Saving preamp: ")); SerialUSB.println(preamp); 
        Settings1.variable=preamp;                // Store default variable setting to EEPROM
        SerialUSB.print(F("Saving preamp_def_vol: ")); SerialUSB.println(preamp_def_vol);
        Settings1.preamp_def_vol=preamp_def_vol;  // Store default preamp volume setting to EEPROM
        SerialUSB.print(F("Saving backlight: ")); SerialUSB.println(backlight);
        Settings1.backlight=backlight;            // Store default TFT backlight setting to EEPROM
        myEEPROM.write(eeAddress, (byte *)&Settings1, sizeof(DAC_Settings));    // write to EEPROM
        SerialUSB.println(F("Settings updated"));
      }
    
  }

// ----------------------------- Function that checks the pressing of the button --------------------------------------------------------------------
/* Returns key values:
 * 0 = momentarily pressed
 * 1 = sustained for 3sec
 * 2 = sustained for 6sec
*/
void buttonPressed()
  {
    if (millis() - keyPrevMillis >= keySampleIntervalMs) 
      {
        keyPrevMillis = millis();
        
        byte currKeyState = digitalRead(keyPin);
        
        if ((prevKeyState == HIGH) && (currKeyState == LOW)) 
          {
            KeyPressCount = 0;
          }
        else if ((prevKeyState == LOW) && (currKeyState == HIGH)) 
          {
            if (KeyPressCount < longKeyPressCountMax && KeyPressCount >= mediumKeyPressCountMin) 
              {
                //SerialUSB.println("Medium");
                key=2;
              }
            else 
              {
                if (KeyPressCount < mediumKeyPressCountMin) 
                  {
                    //SerialUSB.println("short");
                    key=1;
                  }
              }
          }
        else if (currKeyState == LOW) 
          {
            KeyPressCount++;
            if (KeyPressCount >= longKeyPressCountMax) 
              {
                //SerialUSB.println("long");                
                key=3;
              }
          }
        prevKeyState = currKeyState;
        }
  }

// ----------------------------- Function that test reads the Input memory --------------------------------------------------------------------
void testEEPROM()
  {
    // Load Input_1 settings from EEPROM
    eeAddress = eeAddress1;
    byte *pData = (byte*)&Input[0];
    myEEPROM.read(eeAddress, pData, sizeof(DAC_Input));
    SerialUSB.println(F("Test read Input_1 from EEPROM: "));
    SerialUSB.println(Input[0].filter_val);
    SerialUSB.println(Input[0].IIR_val);
    SerialUSB.println(Input[0].DPLLPCM1);
    SerialUSB.println(Input[0].DPLLPCM2);    
    SerialUSB.println(Input[0].DPLLPCM3);
    SerialUSB.println(Input[0].DPLLPCM4);
    SerialUSB.println(Input[0].DPLLPCM5);    
    SerialUSB.println(Input[0].DPLLPCM6);
    SerialUSB.println(Input[0].DPLLPCM7);    
    SerialUSB.println(Input[0].DPLLPCM8);
    SerialUSB.println(Input[0].DPLLPCM9);
    SerialUSB.println(Input[0].DPLLPCM10); 
    SerialUSB.println(Input[0].DPLLPCM11);
    SerialUSB.println(Input[0].DPLLDSD1);
    SerialUSB.println(Input[0].DPLLDSD2);
    SerialUSB.println(Input[0].DPLLDSD3);
    SerialUSB.println(Input[0].DPLLDSD4);
    SerialUSB.println(Input[0].DPLLDSD5);
    SerialUSB.println(Input[0].lockspeed_val);    
    SerialUSB.println(Input[0].dither_val);    
    SerialUSB.println(Input[0].jitter_elim_val);
    SerialUSB.println(Input[0].bypass_osf_val); 
    SerialUSB.println(Input[0].deemph_bypass_val);
    SerialUSB.println(Input[0].auto_deemph_val);
    SerialUSB.println(Input[0].deemph_sel_val);
  
    // Load Input_2 settings from EEPROM
    eeAddress = eeAddress2;
    byte *pData2 = (byte*)&Input[1];
    myEEPROM.read(eeAddress, pData2, sizeof(DAC_Input));
    SerialUSB.println(F("Test read Input_2 from EEPROM: "));
    SerialUSB.println(Input[1].filter_val);
    SerialUSB.println(Input[1].IIR_val);
    SerialUSB.println(Input[1].DPLLPCM1);
    SerialUSB.println(Input[1].DPLLPCM2);    
    SerialUSB.println(Input[1].DPLLPCM3);
    SerialUSB.println(Input[1].DPLLPCM4);
    SerialUSB.println(Input[1].DPLLPCM5);    
    SerialUSB.println(Input[1].DPLLPCM6);
    SerialUSB.println(Input[1].DPLLPCM7);    
    SerialUSB.println(Input[1].DPLLPCM8);
    SerialUSB.println(Input[1].DPLLPCM9);
    SerialUSB.println(Input[1].DPLLPCM10); 
    SerialUSB.println(Input[1].DPLLPCM11);
    SerialUSB.println(Input[1].DPLLDSD1);
    SerialUSB.println(Input[1].DPLLDSD2);
    SerialUSB.println(Input[1].DPLLDSD3);
    SerialUSB.println(Input[1].DPLLDSD4);
    SerialUSB.println(Input[1].DPLLDSD5);
    SerialUSB.println(Input[1].lockspeed_val);    
    SerialUSB.println(Input[1].dither_val);    
    SerialUSB.println(Input[1].jitter_elim_val);
    SerialUSB.println(Input[1].bypass_osf_val); 
    SerialUSB.println(Input[1].deemph_bypass_val);
    SerialUSB.println(Input[1].auto_deemph_val);
    SerialUSB.println(Input[1].deemph_sel_val);

    // Load Input_3 settings from EEPROM
    eeAddress = eeAddress3;
    byte *pData3 = (byte*)&Input[2];
    myEEPROM.read(eeAddress, pData3, sizeof(DAC_Input));
    SerialUSB.println(F("Test read Input_3 from EEPROM: "));
    SerialUSB.println(Input[2].filter_val);
    SerialUSB.println(Input[2].IIR_val);
    SerialUSB.println(Input[2].DPLLPCM1);
    SerialUSB.println(Input[2].DPLLPCM2);    
    SerialUSB.println(Input[2].DPLLPCM3);
    SerialUSB.println(Input[2].DPLLPCM4);
    SerialUSB.println(Input[2].DPLLPCM5);    
    SerialUSB.println(Input[2].DPLLPCM6);
    SerialUSB.println(Input[2].DPLLPCM7);    
    SerialUSB.println(Input[2].DPLLPCM8);
    SerialUSB.println(Input[2].DPLLPCM9);
    SerialUSB.println(Input[2].DPLLPCM10); 
    SerialUSB.println(Input[2].DPLLPCM11);
    SerialUSB.println(Input[2].DPLLDSD1);
    SerialUSB.println(Input[2].DPLLDSD2);
    SerialUSB.println(Input[2].DPLLDSD3);
    SerialUSB.println(Input[2].DPLLDSD4);
    SerialUSB.println(Input[2].DPLLDSD5);
    SerialUSB.println(Input[2].lockspeed_val);    
    SerialUSB.println(Input[2].dither_val);    
    SerialUSB.println(Input[2].jitter_elim_val);
    SerialUSB.println(Input[2].bypass_osf_val); 
    SerialUSB.println(Input[2].deemph_bypass_val);
    SerialUSB.println(Input[2].auto_deemph_val);
    SerialUSB.println(Input[2].deemph_sel_val);
  
    // Load Input_4 settings from EEPROM
    eeAddress = eeAddress4;
    byte *pData4 = (byte*)&Input[3];
    myEEPROM.read(eeAddress, pData4, sizeof(DAC_Input));
    SerialUSB.println(F("Test read Input_4 from EEPROM: "));
    SerialUSB.println(Input[3].filter_val);
    SerialUSB.println(Input[3].IIR_val);
    SerialUSB.println(Input[3].DPLLPCM1);
    SerialUSB.println(Input[3].DPLLPCM2);    
    SerialUSB.println(Input[3].DPLLPCM3);
    SerialUSB.println(Input[3].DPLLPCM4);
    SerialUSB.println(Input[3].DPLLPCM5);    
    SerialUSB.println(Input[3].DPLLPCM6);
    SerialUSB.println(Input[3].DPLLPCM7);    
    SerialUSB.println(Input[3].DPLLPCM8);
    SerialUSB.println(Input[3].DPLLPCM9);
    SerialUSB.println(Input[3].DPLLPCM10); 
    SerialUSB.println(Input[3].DPLLPCM11);
    SerialUSB.println(Input[3].DPLLDSD1);
    SerialUSB.println(Input[3].DPLLDSD2);
    SerialUSB.println(Input[3].DPLLDSD3);
    SerialUSB.println(Input[3].DPLLDSD4);
    SerialUSB.println(Input[3].DPLLDSD5);
    SerialUSB.println(Input[3].lockspeed_val);    
    SerialUSB.println(Input[3].dither_val);    
    SerialUSB.println(Input[3].jitter_elim_val);
    SerialUSB.println(Input[3].bypass_osf_val); 
    SerialUSB.println(Input[3].deemph_bypass_val);
    SerialUSB.println(Input[3].auto_deemph_val);
    SerialUSB.println(Input[3].deemph_sel_val);
  
    // Load Input_5 settings from EEPROM
    eeAddress = eeAddress5;
    byte *pData5 = (byte*)&Input[4];
    myEEPROM.read(eeAddress, pData5, sizeof(DAC_Input));
    SerialUSB.println(F("Test read Input_5 from EEPROM: "));
    SerialUSB.println(Input[4].filter_val);
    SerialUSB.println(Input[4].IIR_val);
    SerialUSB.println(Input[4].DPLLPCM1);
    SerialUSB.println(Input[4].DPLLPCM2);    
    SerialUSB.println(Input[4].DPLLPCM3);
    SerialUSB.println(Input[4].DPLLPCM4);
    SerialUSB.println(Input[4].DPLLPCM5);    
    SerialUSB.println(Input[4].DPLLPCM6);
    SerialUSB.println(Input[4].DPLLPCM7);    
    SerialUSB.println(Input[4].DPLLPCM8);
    SerialUSB.println(Input[4].DPLLPCM9);
    SerialUSB.println(Input[4].DPLLPCM10); 
    SerialUSB.println(Input[4].DPLLPCM11);
    SerialUSB.println(Input[4].DPLLDSD1);
    SerialUSB.println(Input[4].DPLLDSD2);
    SerialUSB.println(Input[4].DPLLDSD3);
    SerialUSB.println(Input[4].DPLLDSD4);
    SerialUSB.println(Input[4].DPLLDSD5);
    SerialUSB.println(Input[4].lockspeed_val);    
    SerialUSB.println(Input[4].dither_val);    
    SerialUSB.println(Input[4].jitter_elim_val);
    SerialUSB.println(Input[4].bypass_osf_val); 
    SerialUSB.println(Input[4].deemph_bypass_val);
    SerialUSB.println(Input[4].auto_deemph_val);
    SerialUSB.println(Input[4].deemph_sel_val);
  
    // Load Input_6 settings from EEPROM
    eeAddress = eeAddress6;
    byte *pData6 = (byte*)&Input[5];
    myEEPROM.read(eeAddress, pData6, sizeof(DAC_Input));
    SerialUSB.println(F("Test read Input_6 from EEPROM: "));
    SerialUSB.println(Input[5].filter_val);
    SerialUSB.println(Input[5].IIR_val);
    SerialUSB.println(Input[5].DPLLPCM1);
    SerialUSB.println(Input[5].DPLLPCM2);    
    SerialUSB.println(Input[5].DPLLPCM3);
    SerialUSB.println(Input[5].DPLLPCM4);
    SerialUSB.println(Input[5].DPLLPCM5);    
    SerialUSB.println(Input[5].DPLLPCM6);
    SerialUSB.println(Input[5].DPLLPCM7);    
    SerialUSB.println(Input[5].DPLLPCM8);
    SerialUSB.println(Input[5].DPLLPCM9);
    SerialUSB.println(Input[5].DPLLPCM10); 
    SerialUSB.println(Input[5].DPLLPCM11);
    SerialUSB.println(Input[5].DPLLDSD1);
    SerialUSB.println(Input[5].DPLLDSD2);
    SerialUSB.println(Input[5].DPLLDSD3);
    SerialUSB.println(Input[5].DPLLDSD4);
    SerialUSB.println(Input[5].DPLLDSD5);
    SerialUSB.println(Input[5].lockspeed_val);    
    SerialUSB.println(Input[5].dither_val);    
    SerialUSB.println(Input[5].jitter_elim_val);
    SerialUSB.println(Input[5].bypass_osf_val); 
    SerialUSB.println(Input[5].deemph_bypass_val);
    SerialUSB.println(Input[5].auto_deemph_val);
    SerialUSB.println(Input[5].deemph_sel_val);
  
    // Load Settings1 from EEPROM
    eeAddress = eeAddressS;
    byte *pSettings = (byte*)&Settings1;
    myEEPROM.read(eeAddress, pSettings, sizeof(DAC_Settings));
  
    SerialUSB.println(F("Test read Settings from EEPROM: "));
    SerialUSB.println(Settings1.variable);
    SerialUSB.println(Settings1.preamp_def_vol);
    SerialUSB.println(Settings1.backlight);
  }
