/**************************************************************************************************
 * TFT HIFIDUINO Code: My adaptation of the HiFiDUINO code so that is displays on a colour TFT.
 *
 * v1.06    01/11/13 : - Compatible with Buffalo 3 and Buffalo 3 SE. Just comment out the relevant statement.
 *                     - Fixed "OS Filt" & "SR disp".. They were not working correctly.
 *                     - Blue select boxes are gone.. they looked quite bad.
 *                     - Some other minor (mainly aesthetic) fixes..
 *
 * v1.03    30/11/13 : - Buffalo IIISE edition. Just 3 inputs: S/PDIF + 2xUSB.
 *
 * v1.00    03/09/13 : - Finished with aesthetics (new fonts, selection grid, etc. First official release.
 *
 * v0.96    25/03/13 : - Changed all pins so as to be compatible with the new Buffalo Shield.
 *
 * v0.95    25/02/13 : - Added IR codes for select, left, right. Everything now works on MEGA & Due! 
 *                       Display still buggy (no functional problems though).
 * 
 * v0.93    17/02/13 : - Changed the rot encoder code. Should be compatible with Due & Mega but only 
 *                       works on the MEGA for the time being.
 *
 * v0.91g   22/12/12 : - Added code to support rotary encoder on Due & added relevant option for selecting 
 *                       either MEGA or Due board (!! still not working!!).
 *                     - Fixed s/pdif frequency display.
 *
 * v0.9     13/12/12 : - Fixed bug in volume control code. Now works as it is supposed to. :)
 *                     - Remote works very well. To do: TFT backlight control.
 *
 * v.0.8.8  10/12/12 : - Powerup and shutdown routines added (still need to configure IR code for power).
 *                     - Aesthetic changes.
 *
 * v.0.8.7  09/12/12 : - Adapted most of HiFiDuino's enhancements to the code. Display is buggy.
 *
 * v.0.8.6  08/12/12 : - Now saves settings into 24LC256 EEPROM instead of built-in EEPROM (beta, seems 
 *                       to be working OK).
 *
 * v.0.8.5  07/12/12 : - Addition of USB input (9th). To do: set as default input.
 *                     - Now fully tested with I2S sources.
 *
 * v.0.8.4  15/08/12 : - Code cleanup, reordering of functions, etc.
 *
 * v.0.8    28/04/12 : - Now supports 240x400 instead of 240x320 TFT.
 *
 * v.0.7    16/04/12 : - Major revision. Now supports UTFT library for 240x320 TFT.
 *
 * v.0.51   11/03/12 : - Contains a few minor bugs but for the most part works.
 *                     - Includes support of source selection on Buffalo III boards.  
 *                     - The Apple remote codes are changed to another device's that I had handy.
 *                     - Has not been tested with I2S sources.
 *
 * The below comments (by HIFIDUINO) still largely apply.
 *
 ***************************************************************************************************/

/***************************************************************************************************
 * HIFIDUINO Code: Supports Buffalo II 80Mhz and 100Mhz, stereo and dual mono. 
 * 
 * Change log:
 * v. B1.0b 01/07/12:- Compatible with Arduino 1.0 (but not backward compatible)
 *                   - Fixed Minor (potential) bugs
 *                   - Two additional settings: Oversampling bypass and jitter eliminator bypass
 * v. B09f 08/24/11: - Dual MONO: Flipped polarity on one side of each dacs to make it the same as
 *                     TPA's dual mono code (outputs on each side of each DAC have opposite polarity)
 * v. B09e 08/23/11: - Added control for DPLL multiplier
 *                   - Turned on automute loopback to mitigate 382.8KHz audio with 80MHz part
 *                   - Dual MONO compability (outputs on each side of each DAC have same polarity)
 * v. B09c 07/01/11: - Tweaked DIM/Ramp-up to be more gradual (longer at higher volumes)
 *                   - Increased name of input from 3 to 6 characters
 *                   - Customizable features: crystal frequency (80 or 100 MHz)
 *                   - Customizable number of inputs (up to 6) each with its own settings
 * v. B09b 06/21/11: - DIM/Ramp-up control (Soft mute) -set at -70db
 *                   - Remote keys are non-repeating by default  
 *                   - All settings configurable for each (labeled) input (no need for defaults)
 *                   - Save settings in EEPROM. Remembers all settings/last setting
 * v. B08a 06/11/11: - Rearranged and simplified the UI to reclaim some real state and improve looks
 *                   - Added adjustment of notch delay and IIR filter selection
 *                   - Optimized and simplified the code some more
 * v. B07d 06/04/11: - Separate tracking of DPLL setting for I2S and SPDIF
 *                   - Fixed dpll setting not updated when switching between SPDIF and I2S
 *                   - Two versions: B07d80 and B07d100 for different frequency crystal
 * v. B07c 06/02/11: - Requires Arduino 0022 version in order to support pulseIn function for remote
 *                   - Moved muting DACs to the beginning of program to minimize the time for full
 *                     vol during power on of the DAC/Arduino when there is an SPDIF signal present
 *                   - Jitter is now always ON (Did not see the need to turn it off). No selection
 *                   - Supports different quantization bit depth: 6,7,8,9 in pseudo and true diff
 *                   - Implemented "DPLL Priming" which goes through all the settings when the DAC
 *                     first locks unto a signal (done once during a power cycle)
 * v. B06e 01/08/11: Supports 32-bit I2S input in accordance to Buffalo II input wiring of SabreDAC
 *                   Rearranged code for default conditions, correct reading of sample rate in I2S
 *                   Allows manual "best DPLL bandwidth" setting. Allows manual selection of I2S or 
 *                   SPDIF. Can set default startup to I2S or SPDIF (in the code)
 *                   This version compatible with Arduino 0021. Fixes a deficiency in pulseIn()
 *                   by redefining the function with the fix.
 * v. B05  11/25/10: Added remote control capability using the Apple Aluminum Remote Control. This
 *                   version enables remote volume control only.
 * v. B04  11/06/10: Added large numbers, tidy-up the UI, "pulse" indicator -every time the Status
 *                   information is read. Also status indicator for PCM/DSD and Signal Lock 
 * v. B03  11/02/10: Supports s/w debounce, added setting of FIR filter, DPLL bandwidth, jitter
 *                   reduction and adjustment of LCD brightness. 
 * v. B021 10/18/10: No new functionality, but cleaned up the code and comments.
 * v. B02  10/15/10: Added reading of sample rate.
 * v. B01  10/11/10: Volume control, LCD, Rotary Encoder.
 ***************************************************************************************************/
/***************************************************************************************************
 * look for "code customization section" to customize code to your setup
 *
 ***************************************************************************************************/
/***************************************************************************************************
 * The following is a description of some of the registers of Sabre32. Based on datasheet and other
 * sources and also my own experimentation
 ***************************************************************************************************/
/*
 1-Set up for I2S/DSD support according to BuffII input wiring
 
 Register 14 (0x0E) DAC source, IIR Bandwidth and FIR roll off
 |0| | | | | | | | Source of DAC8 is DAC8 (D)
 |1| | | | | | | | Source of DAC8 is DAC6 (To Buffalo II wiring)
 | |0| | | | | | | Source of DAC7 is DAC7 (D)
 | |1| | | | | | | Source of DAC7 is DAC5 (To Buffalo II wiring)
 | | |0| | | | | | Source of DAC4 is DAC4 (D)
 | | |1| | | | | | Source of DAC4 is DAC2 (To Buffalo II wiring)
 | | | |0| | | | | Source of DAC3 is DAC3 (D)
 | | | |1| | | | | Source of DAC3 is DAC1 (To Buffalo II wiring)
 | | | | |0| | | | Pseudo differential
 | | | | |1| | | | True differential (D)
 | | | | | |0|0| | IIR Bandwidth: Normal (for PCM)
 | | | | | |0|1| | IIR Bandwidth: 50k (for DSD) (D)
 | | | | | |1|0| | IIR Bandwidth: 60k (for DSD)
 | | | | | |1|1| | IIR Bandwidth: 70k (for DSD)
 | | | | | | | |0| FIR Rolloff: Slow
 | | | | | | | |1| FIR Rolloff: Fast (D)
 
 Buffalo II input pins are wired as such: 1111xxxx for example
 
 11111001 or 0xF9 Default for BII wiring, true differential, IIR normal, FIR fast
 Pseudo diff: clear bit 3
 true diff:   set   bit 3
 Slow FIR:    clear bit 0
 Fast FIr:    set   bit 0
 ------------------ 
 
 2- Set up I2S bit depth. In theory I2S can be set to 32 bit and it
 will automatically handle all bit depths.
 
 Register 10 (0x0A) (MC1)
 |0|0| | | | | | |  24bit data
 |0|1| | | | | | |  20bit data
 |1|0| | | | | | |  16bit data
 |1|1| | | | | | |  32bit data (D)
 | | |0|0| | | | |  I2S (D)
 | | |0|1| | | | |  LJ
 | | |1|0| | | | |  RJ
 | | |1|1| | | | |  I2S (Not sure why two values for I2S)
 | | | | |1| | | |  Reserved for engineering; must be 1 (D)
 | | | | | |0| | |  Jitter Reduction Bypass
 | | | | | |1| | |  Jitter Reduction ON (D)
 | | | | | | |0| |  Deemph ON
 | | | | | | |1| |  Deemph Bypass (D) (can set auto deemph in Reg17)
 | | | | | | | |0|  Unmute DACs (D)
 | | | | | | | |1|  Mute DACs
 
 32 bit I2S, Other defaults such as Jitter ON, Deemphasis Bypass...
 11001111 or 0xCF (Jitter on, etc, MUTE DACs)  (0xFF is also valid)
 11001110 or 0xCE (Unmute DACs)
 ------------------
 
 3- Set up to allow all DPLL settings
 
 Register 25 (0x19): DPLL Mode control
 |0|0|0|0|0|0| | | Reserved, must be zeros (D)
 | | | | | | |0| | DPLL Bandwidht: allow all Settings
 | | | | | | |1| | DPLL Bandwidht: Use best DPLL Settings (D)
 | | | | | | | |0| DPLL128x: Use DPLL setting (D)
 | | | | | | | |1| DPLL128x: Multiply DPLL by 128
 
 Allow all DPLL settings at 1x:
 00000000 or 0x00
 Use best DPLL settings:
 00000010 or 0x02
 Allow all DPLL settings at 128x:
 00000000 or 0x01
 ------------------ 
 
 4- Set up for DPLL bandwidth
 
 Register 11 (0x0B) (MC2)
 |1|0|0| | | | | |  Reserved for engineering, Must be 100 (D)
 | | | |0|0|0| | |  DPLL BW: No Bandwidth
 | | | |0|0|1| | |  DPLL BW: Lowest (D)
 | | | |0|1|0| | |  DPLL BW: Low
 | | | |0|1|1| | |  DPLL BW: Medium-Low 
 | | | |1|0|0| | |  DPLL BW: Medium 
 | | | |1|0|1| | |  DPLL BW: Medium-High
 | | | |1|1|0| | |  DPLL BW: High 
 | | | |1|1|1| | |  DPLL BW: Highest 
 | | | | | | |0|0|  DeEmph: 32k
 | | | | | | |0|1|  DeEmph: 44.1k (D)
 | | | | | | |1|0|  DeEmph: 48k
 | | | | | | |1|1|  Reserved, do not use
 
 Lowest Bandwidth 44.1K De-emphasis select (these are chip default):
 10000101 or 0x85 (or decimal 133)
 ---
 10000001 or 0x81 (No Bandwidth)
 10000101 or 0x85 (lowest)
 10001001 or 0x89 (low)
 10001101 or 0x8D (med-low)
 10010001 or 0x91 (med)
 10010101 or 0x95 (med-hi)
 10011001 or 0x99 (high)
 10011101 or 0x9D (highest)
 ------------------
 
 5- Set up reg 17 for manual or auto  SPDIF selection
 
 Register 17 (0x11) (MC5)
 |1| | | | | | | |  Mono Right (if set for MONO)
 |0| | | | | | | |  Mono Left (if set for MONO) (D)
 | |1| | | | | | |  OSF (Oversample filter) Bypass
 | |0| | | | | | |  Use OSF (D)
 | | |1| | | | | |  Relock Jitter Reduction
 | | |0| | | | | |  Normal Operation Jitter Reduction (D)
 | | | |1| | | | |  SPDIF: Auto deemph ON (D)
 | | | |0| | | | |  SPDIF: Auto deemph OFF
 | | | | |1| | | |  SPDIF Auto (Only if no I2S on pins) (D)
 | | | | |0| | | |  SPDIF Manual (Manually select SPDIF input format)
 | | | | | |1| | |  FIR: 28 coefficients (D)
 | | | | | |0| | |  FIR: 27 coefficients
 | | | | | | |1| |  FIR: Phase invert
 | | | | | | |0| |  FIR: Phase NO invert (D)
 | | | | | | | |1|  All MONO (Then select Mono L or R)
 | | | | | | | |0|  Eight channel (D)
 
 Auto SPDIF, others at defaults:
 00001100 (0x1C)
 Manual SPDIF, others at default:
 00000100 (0x14)
 ------------------
 
 6- Enable I2S/DSD
 
 Register 8 (0x08) Auto-mute level, manual spdif/i2s
 |0| | | | | | | |  Use I2S or DSD (D)
 |1| | | | | | | |  Use SPDIF
 | |1|1|0|1|0|0|0|  Automute trigger point (D)
 
 I2S/DSD input format:
 01101000 (0x68)
 SPDIF input format:
 11101000 (0xE8) // (Actually just disables I2S input format)
 
 If we use a different automute trigger point, say -60 db then
 (This is for experimenting the automute feature)
 I2S/DSD input format:
 00111100 (0x3C)
 SPDIF input format:
 10111100 (0xBC) // (Actually just disables I2S input format)
 
 
 NOTE on auto/manual SPDIF selection
 ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 Apparentely, auto detection of spdif and I2S (with I2S/DSD enabled in reg 8), only works when the
 DAC powers up. Once you turn off auto-SPDIF and then turn on auto-SPDIF, it will not work with SPDIF
 input format while reg 8 is still set for I2S/DSD. In order for SDPIF to work, reg 8 must be set for
 SPDIF and reg 17 for auto-SDPIF. In summary:
 
 reg 17 auto-SPDIF | reg 8 source | Result
 ~~~~~~~~~~~~~~~~~~|~~~~~~~~~~~~~~|~~~~~~~
 ON              | I2S/DSD      | SPDIF ON: Only works when DAC starts with these (default) settings
 ON              | SPDIF        | SPDIF ON: Works even when auto-SPDIF is set to OFF, then to ON (1)
 OFF             | SPDIF        | SPDIF OFF, I2S OFF (3)
 OFF             | I2S/DSD      | I2S ON (2)
 
 Thus for manual operation, use (1) to select SDPIF; use (2) to select I2S. I don't know what is the
 purpose of (3)
 
 Writing to register 18 (selecting the spdif input line) alone will not work if auto-SPDIF is off.
 You need to turn on auto-SPDIF and if using SPDIF input line other than #1, you need to select the
 input line with register 18 (tested may 29, 2011)   
 ------------------ 
 
 7- Reroute SPDIF input line to pin Data5 which is also connected to D1 (Not needed for BII)
 
 Register 18 (0x12): Spdif Input line
 |0|0|0|0|0|0|0|1|  SDPIF Input: Data1 (D) (In BuffII, spdif is connected to this pin)
 |0|0|0|0|0|0|1|0|  SDPIF Input: Data2
 |0|0|0|0|0|1|0|0|  SDPIF Input: Data3
 |0|0|0|0|1|0|0|0|  SDPIF Input: Data4
 |0|0|0|1|0|0|0|0|  SDPIF Input: Data5 (In BuffII, spdif is also  connected to this pin)
 |0|0|1|0|0|0|0|0|  SDPIF Input: Data6
 |0|1|0|0|0|0|0|0|  SDPIF Input: Data7
 |1|0|0|0|0|0|0|0|  SDPIF Input: Data8
 
 Re-route spdif input line to Data5 (Maybe it will help the I2S lock issue):
 00010000 or 0x10
 Re-route spdif input line to Data8 (which is not a valid spdif input, but it is grounded)
 10000000 or 0x80
 ------------------ 
 
 Register 37 (0x25) 
 |0|0| | | | | | |  Reserved (D)
 | | |1| | | | | |  Use downloaded coefficients for stage 1 FIR filter
 | | |0| | | | | |  Use built-in coefficients for stage 1 FIR filter (D)
 | | | |1| | | | |  Enable writing of coefficients for stage 1 -Use when starting to write coeffs
 | | | |0| | | | |  Not enabled to write coefficients for stage 1 (D) -Also use when done writing
 | | | | |0|0| | |  Reserved (D)
 | | | | | | |1| |  Use downloaded coefficients for stage 2 FIR filter
 | | | | | | |0| |  Use built-in coefficients for stage 2 FIR filter (D)
 | | | | | | | |1|  Enable writing of coefficients for stage 2 -Use when starting to write coeffs
 | | | | | | | |0|  Not enabled to write coefficients for stage 2 (D) -Also use when done writing
 
 Start writing coefficients into stage 1:
 00010000 or 0x10; 00110000 or 0x30 might work too
 Done writing coefficients into stage 1:
 00000000 or 0x00
 
 Use built-in coefficients for stage 1 and stage 2:
 00000000 or 0x00
 Use downloaded coefficients for stage 1 and built-in for stage 2:
 00100000 or 0x20
 Use downloaded coefficients for stage 1 and stage 2:
 00100010 or 0x22
 Use built-in coefficients for stage 1 and downloaded for stage 2:
 00000010 or 0x02
 ------------------
 
 Reg 38 (0x26)
 Reg 39 (0x27)
 Reg 40 (0x28)
 Reg 41 (0x29)
 ------------------
 
 Register 15 (0x0F) Quantizer bit depth
 6-bit: 0x00 (D)
 7-bit: 0x55
 8-bit: 0xAA
 9-bit: 0xFF
 ------------------
 
 Register 13 (0x0D) DAC polarity
 In-phase: 0x00 (all 8 channels) (D)
 Anti-phase: 0xFF (all 8 channels)
 
 ------------------
 
 Register 19 (0x13) DACB polarity
 In-phase: 0xFF (all 8 channels)
 Anti-phase: 0x00 (all 8 channels) (D)
 1256 In-Phase; 3478 Anti-Phase: 0x33
 1256 Anti-Phase; 3478 In-Phase: 0xCC
 ------------------
 
 Register 12 (0x0C) 
 |0| | | | | | | | Dither Control: Apply
 |1| | | | | | | | Dither Control: Use fixed rotation pattern
 | |0| | | | | | | Rotator Input: NS-mod input
 | |1| | | | | | | Rotator Input: External input
 | | |0| | | | | | Remapping: No remap
 | | |1| | | | | | Remapping: Remap diigital outputs for max phase separation in analog cell
 
 | | | |0|0|0|0|0| No Notch
 | | | |0|0|0|0|1| Notch at MCLK/4
 | | | |0|0|0|1|1| Notch at MCLK/8
 | | | |0|0|1|1|1| Notch at MCLK/16
 | | | |0|1|1|1|1| Notch at MCLK/32
 | | | |1|1|1|1|1| Notch at MCLK/64
 
 |0|0|1|0|0|0|0|0| Power-on Default
 
 Values to use:
 00100000 or 0x20 NtNO
 00100001 or 0x21 Nt04
 00100011 or 0x23 Nt08
 00100111 or 0x27 Nt16
 00101111 or 0x2F Nt32
 00111111 or 0x3F Nt64
 
 */

/***************************************************************************************************
 * Code starts here
 ***************************************************************************************************/

// LIBRARIES
#include <Wire.h> // For I2C
//#include <EEPROM.h> // We will use the eeprom to store values

// CONSTANT DEFINITION

#include <UTFT.h>

// Declare which fonts we will be using
extern uint8_t SmallFont[];
extern uint8_t BigFont[];
extern uint8_t Ubuntubold[];
extern uint8_t Sinclair_S[];
extern uint8_t arial_bold[];
extern uint8_t DotMatrix_M[];
extern uint8_t Shruti_Bold_num_48x70[];

UTFT myGLCD(ITDB32WD,38,39,40,41);

//******* Bitmaps *********/

//extern unsigned int TP_Logo_200[0x400];
//extern unsigned int dimdim[0x400];

/******************* Code Customization Section *********************/


/* First: Choose the clock frequency you have and comment the other */

//#define USE80MHZ  
#define USE100MHZ

/* Second: Choose your configuration
 
 | CONFIGURATION       | #define DUALMONO | #define TPAPHASE |
 |---------------------|------------------|------------------|
 | Dual mono in-phase  | un-comment       | comment          |
 | Dual mono TPA phase | un-comment       | un-comment       |
 | Stereo              | comment          | comment          |
 |---------------------|------------------|------------------|    */

//#define DUALMONO
//#define TPAPHASE

/* Optionally choose the kind of Arduino board you will be using. Choose the MEGA or the Due.*/
//#define ARDUMEGA  
//#define ARDUDUE

/* Choose the kind of Buffalo that you have. Either BUFFALO3 or B3SE. If you have a 32S or a II choose B3SE.*/
#define BUFFALO3
//#define B3SE

/* Optionally choose whether you will be using the remote power on/off functionality.*/
/* Comment out if you will be using the remote for on/off.*/
//#define ALWAYSON


/* Optionally choose the number of inputs.*/
#ifdef B3SE
#define ICHO 3
#endif B3SE

#ifdef BUFFALO3
#define ICHO 10
#endif BUFFALO3

/* Optionally change the name of the inputs. Use blanks if necessary */
#ifdef B3SE
char no0[] = "S/PDIF ";
char no1[] = "USB 1  ";
char no2[] = "USB 2  ";
#endif B3SE

#ifdef BUFFALO3
char no0[] = "COAX 1 ";
char no1[] = "OPT 1  ";
char no2[] = "OPT 2  ";
char no3[] = "AES/EBU";
char no4[] = "COAX 2 ";
char no5[] = "COAX 3 ";
char no6[] = "COAX 4 ";
char no7[] = "COAX 5 ";
char no8[] = "USB 1  ";
char no9[] = "USB 2  ";
#endif BUFFALO3

/* Make sure you use the correct chip address for each board
 
 for stereo Buffalo: use address 0x90
 for dual mono: Use address 0x90 for mono left Buffalo 
 Use address 0x92 for mono right Buffalo           */

/********************************************************************/

#define DEFAULTATTNU 0x64 //-50 dB this is 50x2=100 or 0x64. Sabre32 is 0 to -127dB in .5dB steps
#define MAXATTNU 0xC6     //-99dB this is 99X2=198 or 0xC6 -Can go higher but LCD shows 2 digits
#define MINATTNU 0x00     //-0 dB -Maximum volume is -0 dB
#define DIM 0x8C          //-70 dB this is 70x2=140 or 0x8C. Dim volume
#define RAMP 10           // Additional msec delay per 1 db for ramping up to volume after dim

int VOLUPPIN=6;          // Button to increase  volume or RotEnc A terminal
int VOLDOWNPIN=7;        // Button to decrease volume or RotEnc B terminal
#define SELECTPIN 5      // Switch to select function

int irdir=0;

// #define BRIPIN 3         // Pin to control the LCD brightness with analogWrite
#define REMOTEPIN 9      // Pin for IR receiver (remote control)

#ifdef B3SE
#define SELECTI2SPIN A7      // Pin for S/PDIF vs I2S selection. High for S/PDIF. Low for I2S.
#define AUX1PIN A2          // Pin #1 for controlling other things
#define AUX2PIN 3          // Pin #2 for controlling other things
#endif B3SE

#ifdef BUFFALO3
#define SELECTI2SPIN A2      // Pin for Sidecar activation (Select I2S vs. S/PDIF)
#define AUX1PIN 3          // Pin #1 for controlling other things
#define AUX2PIN A7          // Pin #2 for controlling other things
#endif BUFFALO3

#define POWERPIN A0        // Pin for main power activation
#define DIMMPIN 8          // Pin for TFT backlight controll. A LOW on this pin turns on the backlight.

#define INTERVAL_SAMPLE 2     // Time interval in SECONDS for refreshing the sample rate
#define INTERVAL_BOUNCE 2     // Time in milliseconds to debounce the rotary encoder
#define INTERVAL_SWITCHBOUNCE 200  // Time in milliseconds to debounce switch
#define INTERVAL_SELECT 4     // Time in sec to exit select mode when no activity

#define VOL 0  // The order of selection when clicking the select switch
#define INP 1  // Input selection
#define INF 2  // Input format selection
#define FIR 3  // FIR filter selection
#define IIR 4  // IIR filter selection
#define DPL 5  // DPLL bandwidth setting
#define QTZ 6  // Quantizer selection
#define NCH 7  // Notch selection
#define PLM 8  // DPLL mode
#define OSF 9  // Oversampling
#define SRF 10 // Sample Rage format: "nominal" or "exact"

// Order of settings in the array for each input
#define FORMATVAL 0
#define FIRVAL 1
#define IIRVAL 2
#define DPLLVAL 3
#define QUANVAL 4
#define NOTCHVAL 5
#define PLMVAL 6
#define OSFVAL 7
#define SRFVAL 8

/* Total number of parameters to keep track for each input. The 7 listed above plus the input
 choice. This makes 8.
 OSF is not tracked  per input as it always starts with oversampling ON
 VOL is also not tracked per input as it always starts with the default volume
 SRF is also not trackec per input as it applies to all inputs
 Thus there are 11 parameters, 8 which are saved per input, and 3 are across all inputs
 */

#define MAXPARAM 10                      

// Number of valid choices for each parameter
#define FORMATCHO 2 // SPDIF, I2S/DSD
#define FIRCHO 2    // Fast, slow
#define IIRCHO 4    // PCM, 50k, 60k, 70k
#define DPLLCHO 8   // Lowest to Best 8 values
#define QUANCHO 6   // 6bT, 7bP, 7bT, 8bP, 8bT, 9bP
#define NOTCHCHO 6  // /4 to /64
#define PLMCHO 4    // Normal x1 (NOR), Multiplier x128 (MUL), Bypass (OFF), INV (Invert phase)
#define OSFCHO 2
#define SRFCHO 2
// There is one more: ICHO defined above

#define chip1 0x50    // device address for the 24LC256 EEPROM chip

// VARIABLE DECLARATION

// Register variables: used for registers that have lots of options. They are initialized here
// with valid values, but will eventually be overwritten by the values in EEPROM
byte reg14=0xF9; // Default value for register 14. We use a variable for reg 14 because it controls
// several parameters: IIR, FIR, differential whereas the other registers typically
// controls a single parameter.
byte reg25=0;    // 0= allow all settings
byte reg17L=0x1C;  // Auto SPDIF, 8-channel mode, other defaults
// reg17L is used for stereo and for left channel if set for MONO
#ifdef DUALMONO
byte reg17R=0x9D;  // Auto SPDIF, MONO RIGHT CHANNEL, other defaults
#endif DUALMONO

byte reg10=0xCE; // jitter ON, dacs unmute, other defaults

unsigned long displayMillis = 0;   // Stores last recorded time for display interval
unsigned long debounceMillis = 0;  // Stores last recorded time for switch debounce interval
unsigned long selectMillis = 0;    // Stores last recorded time for being in select mode
unsigned long sr = 0;              // Stores the incoming sample rate. Used for auto OSF bypass for 80MHz crystal
//unsigned long srold = 0;           // Stores the already displayed incoming sample rate. Used to save screen refreshes if the sr does not change.

byte input=0;                 // The current input to the DAC
byte currAttnu=DEFAULTATTNU;  // Variable to hold the current attenuation value

byte select;          // To record current select position (FIL, VOL, DPLL, etc)

boolean selectMode;   // To indicate whether in select mode or not
boolean SPDIFValid;   // To indicate whether SPDIF valid data
boolean spdifIn;      // To indicate whether in I2S/DSD or SPDIF input format mode
// boolean spdifInold;
boolean bypassOSF=false; // false=no bypass; This is the default condition at startup
boolean SRExact=true;    // Display exact sample rate value; false = display nominal value
boolean primed=false; // To indicate if dpll has been conditioned (once at startup)
boolean dimmed=false; // To indicate dim (mute) or not
boolean poweron=false;  // Default power-on condition: off
byte pulse=0;         // Used by the "heartbeat" display
byte Status;          // To hold value of Sabre32 status register

// The following variables for the remote control feature
int duration;         // Duration of the IR pulse
int mask;             // Used to decode the remote signals 
int c1;               // Byte 1 of the 32-bit remote command code
int c2;               // Byte 2 of the 32-bit remote command code
int c3;               // Byte 3 of the 32-bit remote command code
int c4;               // Byte 4 of the 32-bit remote command code
int IRkey;            // The unique code (Byte 3) of the remote key
int previousIRkey;    // The previous code (used for repeat)


// The array holds the parameters for each input
// The current input is recorded after the array
byte settings[ICHO][MAXPARAM];  // Array to hold parameter values

void writeSettings(){
  for(byte i=0;i<ICHO;i++) {
    for (byte j=0;j<MAXPARAM;j++) {
      if(readData(chip1,(i*MAXPARAM)+j)!=settings[i][j]) // Check an see if there are any changes
        writeData(chip1,(i*MAXPARAM)+j,settings[i][j]);  // Write the changes in eeprom
    }
  }
  writeData(chip1,ICHO*MAXPARAM,input); // Write the current input in a variable location after the array
  writeData(chip1,ICHO*MAXPARAM+1,SRExact); // Write the value of SRExact
}

void readSettings(){
  for(byte i=0;i<ICHO;i++) {
    for (byte j=0;j<MAXPARAM;j++) {
      settings[i][j]=readData(chip1, (i*MAXPARAM)+j);
    }
  }
  input=readData(chip1,ICHO*MAXPARAM);  // Read the last saved input setting
  SRExact=readData(chip1,ICHO*MAXPARAM+1);  // Read the last saved setting for SRExact
}

/*
 ROTARY ENCODER
 
 The rotary encoder is connected to digital pin 2 (A) and digital pin 4 (B). It does not matter 
 which terminal is connected to which pin. The third terminal is connected to GND. At each cycle, 
 the rotary encoder will pull the pins LOW or HIGH.
 
 Debounce
 In this version, the code implements debouncing by adding a few millisecond delay that pauses 
 the code from evaluating the state of the pins. Typically, all the switching noise –the “bouncing” 
 is generated in the first few milliseconds after the switch event. The code is optimized by calling 
 the delay only if there is any activity in the rotary encoder.
 
 There is just a slight change in the software if H/W debouncing is implemented, namely the pull up 
 resistors in the pins are to be disabled.
 
 Further, the rotary encoder has an on-off switch, and the debouncing of the switch is also done in s/w
 (again the same h/w debouncing can be implemented, but it is optional). Because pushing the switch manually 
 will generate not only noise during the switching, but the switch can remain pressed for 100 milliseconds 
 or more because one cannot lift the finger that fast.
 In this implementation and with my way of pusshing down the switch, 200 msec is an appropriate value.
 
 */

// Rotary encoder service routine
static boolean rotating=false;
void rotEncoder()
{
  rotating=true; // If motion is detected in the rotary encoder, set the flag to true
}

static boolean rotating2=false;
void rotEncoder2()
{
  rotating2=true; // If motion is detected in the rotary encoder, set the flag to true
}

/*
READING THE SAMPLE RATE
 
 The sample rate can be calculated by reading the DPLL 32-bit register. For SPDIF DPLL value 
 is divided by (2^32/Crystal-Frequency). In Buffalo II (original), the Crystal frequency is
 80,000,000 Hz. In Arduino (and other small microprocessors) it is NOT advisable to do floating point
 math because "it is very slow"; therefore integer math will be used to calculate the sample rate.
 
 The value of 2^32/80,000,000 is 53.687091 (which is a floating point number). If we use the 
 integer part (53 or 54) we get the following results for a 44.1K sample rate signal:  divided by 53 
 the result is 44.677K; divided by 54, the result is 43.849K. Clearly there are large errors from 
 being confined to integer math. The actual result, if we use floating point math and use all the 
 significant digits is 44,105 Hz (the 5 Hz deviation from ideal 44100 Hz is within the specification
 of SPDIF and the tolerances of the crystals and clocks involved)
 
 In order to increase the accuracy of the integer calculation, we can use more of the significant 
 digits of the divisor. I did some evaluation of the DPLL register values for sample rates ranging 
 from 44.1K to 192K and noticed that I could multiply the value of the DPLL number by up to 
 400 without overflowing the 32-bits. Therefore, since we have 32 bit number to work with, we 
 can multiply the DPLL number by  up to 400 and then divide by 400X53.687091=21475. If we do this, 
 we obtain 44.105K which is the same as the exact value.
 
 I used a spreadsheet to calculate the multipliers for SPDIF and I2S and for both 80Mhz and 100Mhz
 
 SPDIF 80Mhz:  x80, %4295
 SPDIF 100Mhz: x20, %859
 I2S 80Mhz:     x1, %3436
 I2S 100Mhz:    x4, %10995 (higher multiplier will overflow the 32bit value for 384KHz SR)
 x5, %13744 (More accurate but only works up to 192KHz SR)
 
 For I2S input format the dpll number is divided by (2^32*64/Crystal-Frequency) Note the 64 factor.
 The value of this is 3435.97 which rounds off nicely to 3436 (which is only 0.0008% error). The
 resultant value for the sample rate is the same wheter in spdif or I2S mode.
 */

// Sample rate reading routines

volatile unsigned long DPLLNum; // Variable to hold DPLL value
volatile unsigned long RegVal; // Variable to hold Register values

byte readDPLL(byte regAddr) {
  Wire.beginTransmission(0x48); // Hard coded the Sabre/Buffalo device  address
  Wire.write(regAddr);          // Queues the address of the register
  Wire.endTransmission();       // Sends the address of the register
  Wire.requestFrom(0x48,1);     // Hard coded to Buffalo, request one byte from address
  if(Wire.available())          // Wire.available indicates if data is available
      return Wire.read();         // Wire.read() reads the data on the wire
  else
    return 0;                   // In no data in the wire, then return 0 to indicate error
}

unsigned long sampleRate() {
  DPLLNum=0;
  // Reading the 4 registers of DPLL one byte at a time and stuffing into a single 32-bit number
  DPLLNum|=readDPLL(31);
  DPLLNum<<=8;
  DPLLNum|=readDPLL(30);
  DPLLNum<<=8;
  DPLLNum|=readDPLL(29);
  DPLLNum<<=8;
  DPLLNum|=readDPLL(28);
  // The following calculation supports 80 MHz oscillator
  if (SPDIFValid){
#ifdef USE80MHZ
    DPLLNum*=80;      // Calculate SR for SPDIF -80MHz part
    DPLLNum/=4295;    // Calculate SR for SDPIF -80MHz part
#endif
#ifdef USE100MHZ
    DPLLNum*=20;      // Calculate SR for SPDIF -100MHz part 
    DPLLNum/=859;     // Calculate SR for SDPIF -100MHz part
#endif
  }
  else {              // Different calculation for SPDIF and I2S
#ifdef USE80MHZ
    DPLLNum/=3436;    // Calculate SR for I2S -80MHz part
#endif
#ifdef USE100MHZ
    DPLLNum*=4;      // Calculate SR for I2S -100MHz part
    DPLLNum/=10995;  // Calculate SR for I2S -100MHz part
#endif
  }
  if (DPLLNum < 90000) // Adjusting because in integer operation, the residual is truncated
    DPLLNum+=1;
  else
    if (DPLLNum < 190000)
      DPLLNum+=2;
    else
      if (DPLLNum < 350000)
        DPLLNum+=3;
      else
        DPLLNum+=4;

  if(bypassOSF)      // When OSF is bypassed, the magnitude of DPLL is reduced by a factor of 64
    DPLLNum*=64;

  return DPLLNum;
}

// Reading the status register. There is a status register providing the following information:
// 1- dsd or pcm mode
// 1- spdif valid or invalid
// 3- spdif mode enabled or disabled. 
// 4- Jitter Eliminator locked/not locked to incoming signal

byte readStatus() {             // Hardcoded to the status register
  Wire.beginTransmission(0x48); // Hard coded the Sabre/Buffalo device  address
  Wire.write(27);               // Hard coded to status register
  Wire.endTransmission();
  Wire.requestFrom(0x48,1);     // Hard coded to Buffalo, request one byte from address
  if(Wire.available())
    return Wire.read();         // Return the value returned by specified register
  else
    return 0;
}

// Reading the SPDIF status register. There is a status register providing the following information:
// 1- dsd or pcm mode
// 1- spdif valid or invalid
// 3- spdif mode enabled or disabled. 
// 4- Jitter Eliminator locked/not locked to incoming signal
/*
byte readSPDIFStatus() {             // Hardcoded to the status register
  Wire.beginTransmission(0x48); // Hard coded the Sabre/Buffalo device  address
  Wire.write(48);               // Hard coded to status register
  Wire.endTransmission();
  Wire.requestFrom(0x48,1);     // Hard coded to Buffalo, request one byte from address
  if(Wire.available())
    return Wire.read();         // Return the value returned by specified register
  else
    return 0;
}
*/

/*
byte readDARSReg() {             // Hardcoded to the status register
  Wire.beginTransmission(0x48); // Hard coded the Sabre/Buffalo device  address
  Wire.write(52);               // Hard coded to status register
  Wire.endTransmission();
  Wire.requestFrom(0x48,1);     // Hard coded to Buffalo, request one byte from address
  if(Wire.available())
    return Wire.read();         // Return the value returned by specified register
  else
    return 0;
}
*/

/*
CONTROLLING THE DIGITAL ATTENUATION (VOLUME) -and other registers IN THE DAC
 
 The device address of Sabre DAC Datasheet specifies the address as 0x90 which is an 8-bit value.
 The wire library in Arduino uses 7-bit device addresses and the 8th R/W bit is added automatically
 depending on whether you use the write call [beginTransmission()] or the read call [requestFrom()].
 Therefore, you will use the 7 most significant bits of the 8-bit address.
 In our example, 0x90 becomes 0x48 as follows:
 0x90: 10010000 (we eliminate the rightmost bit to get I2C address)
 0x48: 1001000
 When using dual-mono configuration, the other device can be set to addres 0x92
 0x92: 10010010 (we eliminate the rightmost bit to get I2C address)
 0x49: 1001001
 */

void writeSabreReg(byte regAddr, byte regVal)
{
  Wire.beginTransmission(0x48); //Hard coded to the the Sabre/Buffalo device address
  Wire.write(regAddr); // Specifying the address of register
  Wire.write(regVal); // Writing the value into the register
  Wire.endTransmission();

#ifdef DUALMONO
  Wire.beginTransmission(0x49); //Hard coded to the the other Sabre/Buffalo device address
  Wire.write(regAddr); // Specifying the address of register
  Wire.write(regVal); // Writing the value into the register
  Wire.endTransmission();
#endif DUALMONO
}

void setSabreVolume(byte regVal)
{
  writeSabreReg(0, regVal); // set up volume in DAC1
  writeSabreReg(1, regVal); // set up volume in DAC2
  writeSabreReg(2, regVal); // set up volume in DAC3
  writeSabreReg(3, regVal); // set up volume in DAC4
  writeSabreReg(4, regVal); // set up volume in DAC5
  writeSabreReg(5, regVal); // set up volume in DAC6
  writeSabreReg(6, regVal); // set up volume in DAC7
  writeSabreReg(7, regVal); // set up volume in DAC8
}


void dispVol(byte currAttnu)
{
  int x = 220;  // x coordinate of where the volume display area begins
  int y = 95;  // y coordinate of where the volume display area begins

  if (currAttnu/2<10)
  {
    myGLCD.setFont(BigFont);
    myGLCD.setColor(255, 255, 255);
    myGLCD.print("- ", x, y);
    myGLCD.setColor(0, 0, 0);
    myGLCD.fillRect(x+12, y, x+50, y+70);
    myGLCD.setColor(255, 255, 255);
    myGLCD.setFont(Shruti_Bold_num_48x70);
    myGLCD.printNumI(currAttnu/2, x+59, y);
    myGLCD.setFont(Ubuntubold);
    myGLCD.print("dB", x+110, y+25);
  }
  else
  {
    myGLCD.setFont(BigFont);
    myGLCD.setColor(255, 255, 255);
    myGLCD.print("- ", x, y);
    myGLCD.setColor(255, 255, 255);
    myGLCD.setFont(Shruti_Bold_num_48x70);
    myGLCD.printNumI(currAttnu/2, x+11, y);
    myGLCD.setFont(Ubuntubold);
    myGLCD.print("dB", x+110, y+25);
  }
}

void rampUp()
{
  byte i=(DIM-currAttnu); 
  for(byte dimval=DIM;dimval>currAttnu;dimval--){
    setSabreVolume(dimval);
    dispVol(dimval);
    delay((RAMP)*(1+(10/i*i)));
    i--;
  }
}


// Because of register 17 sets MONO/8-channel, different values are written into different chips
void writeSabreLeftReg(byte regAddr, byte regVal)
{
  Wire.beginTransmission(0x48); // Hard coded to the the Sabre/Buffalo device address for stereo
                                // or mono left. For stereo same as writeSabreReg()
  Wire.write(regAddr);          // Specifying the address of register
  Wire.write(regVal);           // Writing the value into the register
  Wire.endTransmission();
}

#ifdef DUALMONO
void writeSabreRightReg(byte regAddr, byte regVal)
{
  Wire.beginTransmission(0x49); //Hard coded to the the Sabre/Buffalo device address
  Wire.write(regAddr); // Specifying the address of register
  Wire.write(regVal); // Writing the value into the register
  Wire.endTransmission();
}
#endif DUALMONO

/*
 The following function prints a bar at the left most column to indicate that we are in "select"
 mode.
 */
void printSelectBar(){

  myGLCD.setColor(0, 150, 255);
  myGLCD.drawRect(198, 23, 380, 90); // Input
  myGLCD.drawRect(199, 24, 379, 89); // Input

  myGLCD.drawRect(0, 0, 230, 19); // Input format
  myGLCD.drawRect(1, 1, 229, 18); // Input format

  myGLCD.drawRect(0, 19, 162, 57); // FIR
  myGLCD.drawRect(1, 20, 161, 56); // FIR

  myGLCD.drawRect(0, 57, 162, 76); // IIR
  myGLCD.drawRect(1, 58, 161, 75); // IIR

  myGLCD.drawRect(0, 76, 162, 114); // DPLL
  myGLCD.drawRect(1, 77, 161, 113); // DPLL

  myGLCD.drawRect(0, 114, 162, 133); // Qtz
  myGLCD.drawRect(1, 115, 161, 132); // Qtz

  myGLCD.drawRect(0, 133, 162, 152); // Notch
  myGLCD.drawRect(1, 134, 161, 151); // Notch

  myGLCD.drawRect(0, 152, 162, 190); // DPLL Mode
  myGLCD.drawRect(1, 153, 161, 189); // DPLL Mode

  myGLCD.drawRect(0, 190, 195, 209); // Oversampling
  myGLCD.drawRect(1, 191, 194, 208); // Oversampling

  myGLCD.drawRect(0, 209, 195, 228); // SR
  myGLCD.drawRect(1, 210, 194, 227); // SR
}

/*
 ..and this one removes said bar.
 */

void removeSelectBar(){

  myGLCD.setColor(0, 0, 0);
  myGLCD.drawRect(198, 23, 380, 90); // Input
  myGLCD.drawRect(199, 24, 379, 89); // Input

  myGLCD.drawRect(0, 0, 230, 19); // Input format
  myGLCD.drawRect(1, 1, 229, 18); // Input format

  myGLCD.drawRect(0, 19, 162, 57); // FIR
  myGLCD.drawRect(1, 20, 161, 56); // FIR

  myGLCD.drawRect(0, 57, 162, 76); // IIR
  myGLCD.drawRect(1, 58, 161, 75); // IIR

  myGLCD.drawRect(0, 76, 162, 114); // DPLL
  myGLCD.drawRect(1, 77, 161, 113); // DPLL

  myGLCD.drawRect(0, 114, 162, 133); // Qtz
  myGLCD.drawRect(1, 115, 161, 132); // Qtz

  myGLCD.drawRect(0, 133, 162, 152); // Notch
  myGLCD.drawRect(1, 134, 161, 151); // Notch

  myGLCD.drawRect(0, 152, 162, 190); // DPLL Mode
  myGLCD.drawRect(1, 153, 161, 189); // DPLL Mode

  myGLCD.drawRect(0, 190, 195, 209); // Oversampling
  myGLCD.drawRect(1, 191, 194, 208); // Oversampling

  myGLCD.drawRect(0, 209, 195, 228); // SR
  myGLCD.drawRect(1, 210, 194, 227); // SR
}

void setAndPrintInputFormat(byte value){
  // This register also controls mono-8channel operation, thus more code...

  myGLCD.setFont(arial_bold);
  myGLCD.setColor(0, 255, 0);
  myGLCD.print("Format: ", 2, 2);

  switch(value){
  case 0:
    writeSabreReg(0x08,0xE8);        // Reg 8: Enable SPDIF input format
    bitSet(reg17L,3);                // Reg 17: auto spdif detection ON -Set bit 3
    writeSabreLeftReg(0x11,reg17L);  // Reg 17: write value into register
#ifdef DUALMONO
    bitSet(reg17R,3);                // Reg 17: auto spdif detection ON -Set bit 3
    writeSabreRightReg(0x11,reg17R); // Reg 17: write value into register
#endif DUALMONO
    spdifIn=true;                    // Indicates input format is spdif. For label for input format
    myGLCD.setColor(0, 255, 0);
    myGLCD.print("S/PDIF ", 112, 2);
    break;

  case 1:
    bitClear(reg17L,3);              // Reg 17: manual SPDIF -Clear bit 3
    writeSabreLeftReg(0x11,reg17L);  // Reg 17: Auto spdif detection OFF
#ifdef DUALMONO
    bitClear(reg17R,3);              // Reg 17: manual SPDIF -Clear bit 3
    writeSabreRightReg(0x11,reg17R); // Reg 17: Auto spdif detection OFF
#endif DUALMONO
    writeSabreReg(0x08,0x68);        // Reg 8: Enable I2S/DSD input format
    spdifIn=false;                   // Set variable to indicate input format is I2S/DSD mode
    myGLCD.setColor(0, 255, 0);
    myGLCD.print("I2S/DSD", 112, 2);   
    break;
  }
}


void setAndPrintFirFilter(byte value){

  myGLCD.setFont(arial_bold);
  myGLCD.setColor(0, 150, 255);
  myGLCD.print("Filter:", 2, 21);

  switch(value){
  case 0:
    bitSet(reg14,0);           // Set bit 0 of reg 14 for sharp fir
    writeSabreReg(0x0E,reg14);
    myGLCD.print("Sharp FIR", 2, 40);
    break;

  case 1:
    bitClear(reg14,0);         // Clear bit 0 of reg 14 for slow fir
    writeSabreReg(0x0E,reg14);
    myGLCD.print("Slow FIR ", 2, 40);
    break;
  }
}

void setAndPrintIirFilter(byte value){

  myGLCD.setFont(arial_bold);
  myGLCD.setColor(0, 150, 255);
  myGLCD.print("IIR: ", 2, 59);

  switch(value){
  case 0:                        // | | | | | |0|0| | IIR Bandwidth: Normal (for PCM)
    bitClear(reg14,1);           // Clear bit 1
    bitClear(reg14,2);           // Clear bit 2
    writeSabreReg(0x0E,reg14);
    myGLCD.print("PCM", 70, 59);
    break;

  case 1:                        // | | | | | |0|1| | IIR Bandwidth: 50k (for DSD) (D)
    bitSet(reg14,1);             // Set bit 1
    bitClear(reg14,2);           // Clear bit 2
    writeSabreReg(0x0E,reg14);
    myGLCD.print("50K", 70, 59);
    break;

  case 2:                        // | | | | | |1|0| | IIR Bandwidth: 60k (for DSD)
    bitSet(reg14,2);             // Set bit 2
    bitClear(reg14,1);           // Clear bit 1
    writeSabreReg(0x0E,reg14);
    myGLCD.print("60K", 70, 59);
    break;

  case 3:                        // | | | | | |1|1| | IIR Bandwidth: 70k (for DSD)
    bitSet(reg14,1);             // Set bit 1
    bitSet(reg14,2);             // Set bit 2
    writeSabreReg(0x0E,reg14);
    myGLCD.print("70K", 70, 59);
    break;
  }
}


void setAndPrintDPLL(byte value){

  myGLCD.setFont(arial_bold);
  myGLCD.setColor(0, 150, 255);
  myGLCD.print("DPLL B/W:", 2, 78);

  switch(value){
  case 0:
    bitSet(reg25,1);            // Reg 25: set bit 1 for "use best dpll"
    writeSabreReg(0x19,reg25);  // Write value into reg 25 for best dpll
    myGLCD.print("Auto    ", 2, 97);
    break;
  case 1:
    bitClear(reg25,1);          // Reg 25: Clear bit 1 for "use all settings"
    writeSabreReg(0x19,reg25);  // Write value into reg 25 for all settings
    writeSabreReg(0x0B,0x85);   // Reg 11: Set corresponding DPLL bandwidth
    myGLCD.print("Lowest  ", 2, 97);
    break;
  case 2:
    bitClear(reg25,1);          // Reg 25: Clear bit 1 for "use all settings"
    writeSabreReg(0x19,reg25);  // Write value into reg 25 for all settings
    writeSabreReg(0x0B,0x89);   // Reg 11: Set corresponding DPLL bandwidth
    myGLCD.print("Low    ", 2, 97);
    break;
  case 3:
    bitClear(reg25,1);          // Reg 25: Clear bit 1 for "use all settings"
    writeSabreReg(0x19,reg25);  // Write value into reg 25 for all settings
    writeSabreReg(0x0B,0x8D);   // Reg 11: Set corresponding DPLL bandwidth
    myGLCD.print("Low-Med  ", 2, 97);
    break;
  case 4:
    bitClear(reg25,1);          // Reg 25: Clear bit 1 for "use all settings"
    writeSabreReg(0x19,reg25);  // Write value into reg 25 for all settings
    writeSabreReg(0x0B,0x91);   // Reg 11: Set corresponding DPLL bandwidth
    myGLCD.print("Medium  ", 2, 97);
    break;
  case 5:
    bitClear(reg25,1);          // Reg 25: Clear bit 1 for "use all settings"
    writeSabreReg(0x19,reg25);  // Write value into reg 25 for all settings
    writeSabreReg(0x0B,0x95);   // Reg 11: Set corresponding DPLL bandwidth
    myGLCD.print("Med-High", 2, 97);
    break;
  case 6:
    bitClear(reg25,1);          // Reg 25: Clear bit 1 for "use all settings"
    writeSabreReg(0x19,reg25);  // Write value into reg 25 for all settings
    writeSabreReg(0x0B,0x99);   // Reg 11: Set corresponding DPLL bandwidth
    myGLCD.print("High    ", 2, 97);
    break;
  case 7:
    bitClear(reg25,1);          // Reg 25: Clear bit 1 for "use all settings"
    writeSabreReg(0x19,reg25);  // Write value into reg 25 for all settings
    writeSabreReg(0x0B,0x9D);   // Reg 11: Set corresponding DPLL bandwidth
    myGLCD.print("Highest ", 2, 97);
    break;
  }
}


void setAndPrintQuantizer(byte value){

  myGLCD.setFont(arial_bold);
  myGLCD.setColor(0, 150, 255);
  myGLCD.print("Qtz: ", 2, 116);

  switch(value){
  case 0:                        // 6-bit true diff
    bitSet(reg14,3);             // True differential
    writeSabreReg(0x0E,reg14);
    writeSabreReg(0x0F,0x00);    // 6-bit quantizer
    myGLCD.print("6bT", 62, 116);
    break;

  case 1:                        // 7-bit pseudo diff
    bitClear(reg14,3);           // Pseudo diff
    writeSabreReg(0x0E,reg14);
    writeSabreReg(0x0F,0x55);    // 7-bit quantizer
    myGLCD.print("7bP", 62, 116);
    break;

  case 2:                        // 7-it true diff
    bitSet(reg14,3);             // True differential
    writeSabreReg(0x0E,reg14);
    writeSabreReg(0x0F,0x55);    // 7-bit quantizer
    myGLCD.print("7bT", 62, 116);
    break;

  case 3:                        // 8-bit pseudo diff
    bitClear(reg14,3);           // Pseudo diff
    writeSabreReg(0x0E,reg14);
    writeSabreReg(0x0F,0xAA);    // 8-bit quantizer
    myGLCD.print("8bP", 62, 116);
    break;  

  case 4:                        // 8-bit true diff
    bitSet(reg14,3);             // True differential
    writeSabreReg(0x0E,reg14);
    writeSabreReg(0x0F,0xAA);    // 8-bit quantizer
    myGLCD.print("8bT", 62, 116);
    break;

  case 5:                        // 9-bit pseudo diff
    bitClear(reg14,3);           // Pseudo diff
    writeSabreReg(0x0E,reg14);
    writeSabreReg(0x0F,0xFF);    // 9-bit quantizer
    myGLCD.print("9bP", 62, 116);
    break; 
  }
}

void setAndPrintNotch(byte value){

  myGLCD.setFont(arial_bold);
  myGLCD.setColor(0, 150, 255);
  myGLCD.print("Notch:", 2, 135);

  switch(value){
  case 0:
    writeSabreReg(0x0C,0x20);    // No notch delay
    myGLCD.print("NON", 97, 135);
    break;
  case 1:
    writeSabreReg(0x0C,0x21);    // notch delay=mclk/4
    myGLCD.print("/04", 97, 135);
    break;
  case 2:
    writeSabreReg(0x0C,0x23);    // notch delay=mclk/8
    myGLCD.print("/08", 97, 135);
    break;
  case 3:
    writeSabreReg(0x0C,0x27);    // notch delay=mclk/16
    myGLCD.print("/16", 97, 135);
    break;  
  case 4:
    writeSabreReg(0x0C,0x2F);    // notch delay=mclk/32
    myGLCD.print("/32", 97, 135);
    break;
  case 5:
    writeSabreReg(0x0C,0x3F);    // notch delay=mclk/64
    myGLCD.print("/64", 97, 135);
    break; 
  }
}


void setAndPrintDPLLMode(byte value){ // Set the DPLL Mode

  myGLCD.setFont(arial_bold);
  myGLCD.setColor(0, 150, 255);
  myGLCD.print("DPLL Mode:", 2, 154);

  switch(value){
  case 0:
    bitSet(reg10,2);           // Set bit 2 of reg 10: jitter reduction ON
    writeSabreReg(0x0A,reg10);
    bitSet(reg25,0);           // Set bit 0 of reg 25 for x128 DPLL bandwidth
    writeSabreReg(0x19,reg25);
    bitClear(reg17L,1);        // Reg 17: Clear bit 1 to NOT invert DPLL phase
    writeSabreLeftReg(0x11,reg17L);
#ifdef DUALMONO
    bitClear(reg17R,1);        // Reg 17: Clear bit 1 to NOT invert DPLL phase -dual mono 
    writeSabreRightReg(0x11,reg17R);
#endif DUALMONO
    myGLCD.print("x128  ", 2, 173);
    break;
  case 1:
    bitSet(reg10,2);           // Set bit 2 of reg 10: jitter reduction ON
    writeSabreReg(0x0A,reg10);
    bitClear(reg25,0);         // Clear bit 0 of reg 25 for x1 DPLL bandwidth
    writeSabreReg(0x19,reg25);
    bitClear(reg17L,1);        // Reg 17: Clear bit 1 to NOT invert DPLL phase
    writeSabreLeftReg(0x11,reg17L);
#ifdef DUALMONO
    bitClear(reg17R,1);        // Reg 17: Clear bit 1 to NOT invert DPLL phase -dual mono 
    writeSabreRightReg(0x11,reg17R);
#endif DUALMONO
    myGLCD.print("x1    ", 2, 173);
    break;
  case 2:                      
    bitSet(reg10,2);           // Set bit 2 of reg 10: jitter reduction ON
    writeSabreReg(0x0A,reg10);
    bitClear(reg25,0);         // Clear bit 0 of reg 25 for x1 DPLL bandwidth
    writeSabreReg(0x19,reg25);
    bitSet(reg17L,1);          // Reg 17: Set bit 1 to invert DPLL phase
    writeSabreLeftReg(0x11,reg17L);
#ifdef DUALMONO
    bitSet(reg17R,1);          // Reg 17: Set bit 1 to invert DPLL phase -dual mono 
    writeSabreRightReg(0x11,reg17R);
#endif DUALMONO
    myGLCD.print("x1 INV", 2, 173);
    break;
  case 3:
    bitClear(reg10,3);         // Clear bit 3 of reg 10: jitter reduction bypass
    writeSabreReg(0x0A,reg10);
    myGLCD.print("None  ", 2, 173);
    break;
  }
}

void setAndPrintBypassOSF(byte value){         // Set and print Bypass OSF
    
    myGLCD.setFont(arial_bold);
    myGLCD.setColor(0, 150, 255);
  
  switch(value){
  case 0:
    bypassOSF=false;
    bitClear(reg17L,6);              // Reg 17: clear bypass oversampling bit in register
    writeSabreLeftReg(0x11,reg17L);  // Reg 17: bypass OSF off
#ifdef DUALMONO
    bitClear(reg17R,6);              // Reg 17: clear bypass oversampling bit in register -dual mono
    writeSabreRightReg(0x11,reg17R); // Reg 17: bypass OSF off
#endif DUALMONO
    myGLCD.print("OS filt:", 2, 192);
    myGLCD.print("On ", 128, 192);      // Indicate oversampling is ON
    break;
  case 1:
    bypassOSF=true;                  // This is to bypass oversampling
    bitSet(reg17L,6);                // Reg 17: set bypass oversampling bit in register
    bitSet (reg17L,5);               // Reg 17: set Jitter lock bit, normal operation
    writeSabreLeftReg(0x11,reg17L);  // Reg 17: bypass OSF on, force relock
    delay(50);
    bitClear(reg17L,5);              // Reg 17: clear relock jitter for normal operation
    writeSabreLeftReg(0x11,reg17L);  // Reg 17: Jitter eliminator Normal operation 
#ifdef DUALMONO
    bitSet(reg17R,6);                // Reg 17: set bypass oversampling bit in register -dual mono
    bitSet (reg17R,5);               // Reg 17: set Jitter lock bit, normal operation -dual mono
    writeSabreRightReg(0x11,reg17R); // Reg 17: bypass OSF on, force relock
    delay(50);
    bitClear(reg17L,5);              // Reg 17: clear relock jitter for normal operation
    writeSabreLeftReg(0x11,reg17L);  // Reg 17: Jitter eliminator Normal operation
#endif DUALMONO
    myGLCD.print("OS filt:", 2, 192);
    myGLCD.print("Off", 128, 192);      // Indicate oversampling is OFF
    break;
  }
}

void setAndPrintSRFormat(byte value){          // This is a toggle function for selecting SR display format
    myGLCD.setFont(arial_bold);
    myGLCD.setColor(0, 150, 255);

  switch(value){
  case 0:
    SRExact=false;                   // Set to Nominal
    myGLCD.print("SR Disp:", 2, 211);
    myGLCD.print("Nom ", 128, 211);    // Indicate NOMINAL mode
    break;

  case 1:
    SRExact=true;                    // Set to display exact sample rate
    myGLCD.print("SR Disp:", 2, 211);
    myGLCD.print("Exct", 128, 211);    // Indicate EXACT mode
    break;
  }
}


void setAndPrintInput(byte value){
  setAndPrintInputFormat(settings[input][FORMATVAL]%FORMATCHO);  // Setup input format value
  setAndPrintFirFilter(settings[input][FIRVAL]%FIRCHO);          // Setup FIR filter value
  setAndPrintIirFilter(settings[input][IIRVAL]%IIRCHO);          // Setup IIR filter value
  setAndPrintDPLL(settings[input][DPLLVAL]%DPLLCHO);             // Setup the DPLL value
  setAndPrintQuantizer(settings[input][QUANVAL]%QUANCHO);        // Setup quantizer value
  setAndPrintNotch(settings[input][NOTCHVAL]%NOTCHCHO);          // Setup notch delay value
  setAndPrintDPLLMode(settings[input][PLMVAL]%PLMCHO);           // Setup dpll mode value
  setAndPrintBypassOSF(settings[input][OSFVAL]%OSFCHO);          // Setup oversampling bypass value
  setAndPrintSRFormat(settings[input][SRFVAL]%SRFCHO);           // Setup Sampling Rate display format

  myGLCD.setFont(Ubuntubold);
  myGLCD.setColor(255, 255, 255);
  myGLCD.print("Input:", 200, 25);

  #ifdef B3SE
  switch (value){
  case 0:
    writeSabreReg(0x12,0x01);          // Set input to SPDIF (#1)
    digitalWrite(SELECTI2SPIN, LOW);   // Keep IP_S high.
    myGLCD.print(no0, 200, 57);
    break;
  case 1:
    writeSabreReg(0x12,0x01);          // Set input to USB 1 (#2)
    digitalWrite(SELECTI2SPIN, HIGH);  // Pull IP_S low.
    digitalWrite(AUX1PIN, LOW);        // Select Input 1 on OTTO-II.
    myGLCD.print(no1, 200, 57);
    break;
  case 2:
    writeSabreReg(0x12,0x01);          // Set input to USB 2 (#3)
    digitalWrite(SELECTI2SPIN, HIGH);  // Pull IP_S low.
    digitalWrite(AUX1PIN, HIGH);       // Select Input 2 on OTTO-II.
    myGLCD.print(no2, 200, 57);
    break;
  }
  #endif B3SE
  
  #ifdef BUFFALO3
  switch (value){
  case 0:
    writeSabreReg(0x12,0x01);        // Set SPDIF to input #1
    digitalWrite(SELECTI2SPIN, HIGH);   // Power off the sidecar relay.
    myGLCD.print(no0, 200, 57);
    break;
  case 1:
    writeSabreReg(0x12,0x02);        // Set SPDIF to input #2
    digitalWrite(SELECTI2SPIN, HIGH);   // Power off the sidecar relay.
    myGLCD.print(no1, 200, 57);
    break;
  case 2:
    writeSabreReg(0x12,0x04);        // Set SPDIF to input #3
    digitalWrite(SELECTI2SPIN, HIGH);   // Power off the sidecar relay.
    myGLCD.print(no2, 200, 57);
    break;
  case 3:
    writeSabreReg(0x12,0x08);        // Set SPDIF to input #4
    digitalWrite(SELECTI2SPIN, HIGH);   // Power off the sidecar relay.
    myGLCD.print(no3, 200, 57);
    break;
  case 4:
    writeSabreReg(0x12,0x10);        // Set SPDIF to input #5
    digitalWrite(SELECTI2SPIN, HIGH);   // Power off the sidecar relay.
    myGLCD.print(no4, 200, 57);
    break;
  case 5:
    writeSabreReg(0x12,0x20);        // Set SPDIF to input #6
    digitalWrite(SELECTI2SPIN, HIGH);   // Power off the sidecar relay.
    myGLCD.print(no5, 200, 57);
    break;
  case 6:
    writeSabreReg(0x12,0x40);        // Set SPDIF to input #7
    digitalWrite(SELECTI2SPIN, HIGH);   // Power off the sidecar relay.
    myGLCD.print(no6, 200, 57);
    break;
  case 7:
    writeSabreReg(0x12,0x80);        // Set SPDIF to input #8
    digitalWrite(SELECTI2SPIN, HIGH);   // Power off the sidecar relay.
    myGLCD.print(no7, 200, 57);
    break;
  case 8:
    writeSabreReg(0x12,0x01);        // Set input to USB 1 (#9)
    digitalWrite(SELECTI2SPIN, LOW);  // Power on the sidecar relay.
    myGLCD.print(no8, 200, 57);
    break;
  case 9:
    writeSabreReg(0x12,0x01);        // Set input to USB 2 (#9)
    digitalWrite(SELECTI2SPIN, LOW);  // Power on the sidecar relay.
    myGLCD.print(no9, 200, 57);
    break;
  }
  #endif BUFFALO3
}

/*
The following function returns the code from the Apple Aluminum remote control. The Apple remote is
 based on the NEC infrared remote protocol. Of the 32 bits (4 bytes) coded in the protocol, only the
 third byte corresponds to the keys. The function also handles errors due to noise (returns 255) and
 the repeat code (returs zero)
 
 The Apple remote returns the following codes:
 
 Up key:     238 135 011 089
 Down key:   238 135 013 089
 Left key:   238 135 008 089
 Right key:  238 135 007 089
 Center key: 238 135 093 089 followed by 238 135 004 089 (See blog for why there are two codes)
 Menu key:   238 135 002 089
 Play key:   238 135 094 089 followed by 238 135 004 089
 
 In this version of the code the Apple remote is not used. Instead, I am using an old remote that also uses the NEC protocol.
 
 */

int getIRkey() {
  c1=0;
  c2=0;
  c3=0;
  c4=0;
  duration=1;
  while((duration=pulseIn(REMOTEPIN, HIGH, 15000)) < 2000 && duration!=0)
  {
    // Wait for start pulse
  }
  if (duration == 0)         // This is an error no start or end of pulse
    return(255);             // Use 255 as Error

  else if (duration<3000)    // This is the repeat
    return (0);              // Use zero as the repeat code

  else if (duration<5000){   // This is the command get the 4 byte
    mask = 1;				    
    for (int i = 0; i < 8; i++){	             // get 8 bits
      if(pulseIn(REMOTEPIN, HIGH, 3000)>1000)        // If "1" pulse
        c1 |= mask;			             // Put the "1" in position
      mask <<= 1;				     // shift mask to next bit
    }
    mask = 1;				    
    for (int i = 0; i < 8; i++){	             // get 8 bits
      if(pulseIn(REMOTEPIN, HIGH, 3000)>1000)        // If "1" pulse
        c2 |= mask;			             // Put the "1" in position
      mask <<= 1;				     // shift mask to next bit
    }
    mask = 1;				    
    for (int i = 0; i < 8; i++){	             // get 8 bits
      if(pulseIn(REMOTEPIN, HIGH, 3000)>1000)        // If "1" pulse
        c3 |= mask;			             // Put the "1" in position
      mask <<= 1;				     // shift mask to next bit
    }
    mask = 1;				    
    for (int i = 0; i < 8; i++){	             // get 8 bits
      if(pulseIn(REMOTEPIN, HIGH, 3000)>1000)        // If "1" pulse
        c4 |= mask;			             // Put the "1" in position
      mask <<= 1;				     // shift mask to next bit
    }

    return(c3);
  }
}

// The following 2 routines set up the registers in the DAC at startup

void startDac1(){               // Set registers not requiring display info

// Print the welcome message and other labels to the LCD

//myGLCD.drawBitmap (1, 1, 200, 41, TP_Logo_200);

  myGLCD.clrScr();
  myGLCD.setFont(BigFont);
  myGLCD.setColor(255, 255, 0);
  myGLCD.print("T F T", CENTER, 55);
  myGLCD.print("H I F I D U I N O", CENTER, 85);
  myGLCD.print("v.1.06 STEREO", CENTER, 200);

#ifdef DUALMONO
  myGLCD.setFont(BigFont);
  myGLCD.setColor(255, 255, 0);
  myGLCD.print("DUALMONO", CENTER, 100);

#endif DUALMONO

#ifdef TPAPHASE
  myGLCD.setFont(BigFont);
  myGLCD.setColor(255, 255, 0);
  myGLCD.print("DUAL-TPA", CENTER, 100);

#endif TPAPHASE

  delay(1500);

  myGLCD.clrScr();

  bitSet(reg10,0);              // Set bit zero for reg 10: Mute DACs
  writeSabreReg(0x0A,reg10);    // Mute DACs. Earliest we can mute the DACs to avoid full vol
  setSabreVolume(currAttnu);    // Reg 0 to Reg 7 Set volume registers with startup volume level
  writeSabreReg(0x0D,0x00);     // DAC in-phase
  writeSabreReg(0x13,0x00);     // DACB anti-phase
  writeSabreReg(0x25,0x00);     // Use built in filter for stage 1 and stage 2
  writeSabreReg(0x0E,reg14);    // Reg 14: BuffII input map, trueDiff, normal IIR and Fast rolloff

#ifdef DUALMONO                // DAC registers default to stereo. Set to MONO L/R for dual MONO
  bitSet(reg17L,0);              // Set for MONO left channel. Right ch variable is already set for MONO
  writeSabreLeftReg(0x11,reg17L);
  writeSabreRightReg(0x11,reg17R);
#endif DUALMONO

#ifdef TPAPHASE  
  /* The outputs on each side of each MONO board will be in opposite phase. In order to do this
   the phase of the odd dacs are out of phase with the even dacs. Further, buffalo is configured
   such that (In dual mono mode) the channel on the DAC which is opposite of the selected channel  
   carries the same analog signal but in anti-phase (odd dacs are the left channel;
   even dacs are the right channel)
   See http://hifiduino.wordpress.com/sabre32/ for further explaination
   */
  writeSabreLeftReg(0x0D,0x22);  // MONO LEFT DACx: odd dacs=in-phase; even dacs=anti-phase
  // writeSabreLeftReg(0x13,0x00);  // MONO LEFT DACBx: all dacs anti-phase with respect to DACx
  writeSabreRightReg(0x0D,0x11); // MONO RIGHT DACx: odd dacs=anti-phase; even dacs=in-phase
  // writeSabreRightReg(0x13,0x00); // MONO RIGHT DACBx: all dacs anti-phase with respect to DACx
#endif TPAPHASE
}

void startDac2(){
  readSettings();
  setAndPrintInput(input);
  dispVol(currAttnu);
  bitClear(reg10,0);                 // Clear bit zero of reg 10: unmute DACs
  writeSabreReg(0x0A,reg10);         // Unmute DACs
}


/************************ MAIN PROGRAM ************************************************************/

void setup() {

  myGLCD.InitLCD();
  myGLCD.clrScr();
  myGLCD.setBackColor(0, 0, 0);

  // Serial.begin(9600);  // This is here for debugging

  // Set up the pin modes
  pinMode(VOLUPPIN, INPUT);       // Button switch or Encoder pin for volume up
  digitalWrite(VOLUPPIN, HIGH);   // If H/W debouncing is implemented, set to LOW

  pinMode(VOLDOWNPIN, INPUT);     // Button switch or Encoder pin for volume down
  digitalWrite(VOLDOWNPIN, HIGH); // If H/W debouncing is implemented, set to LOW

  pinMode(REMOTEPIN, INPUT);      // Pin for IR sensor
  digitalWrite(REMOTEPIN, HIGH);  // Enable pull-up resistor

  pinMode(SELECTPIN, INPUT);      // Button switch or pin for encoder built-in switch 
  digitalWrite(SELECTPIN, HIGH);  // Enable pull-up resistor 

  pinMode(SELECTI2SPIN, OUTPUT);  // Sidecar or IP_S control pin. 
  digitalWrite(SELECTI2SPIN, LOW);//  

  pinMode(AUX1PIN, OUTPUT);       //  
  digitalWrite(AUX1PIN, LOW);     //  

  pinMode(AUX2PIN, OUTPUT);       //  
  digitalWrite(AUX2PIN, LOW);     //  

  pinMode(POWERPIN, OUTPUT);      // Main power control pin. High output powers on the DAC. 
  digitalWrite(POWERPIN, LOW);    // By default, keep power off. 

  pinMode(DIMMPIN, OUTPUT);       // TFT backlight control pin. Low output powers on the backlight. 
  digitalWrite(DIMMPIN, HIGH);    // By default, keep backlight off. 

  #ifdef ALWAYSON
  digitalWrite(POWERPIN, HIGH);
  digitalWrite(DIMMPIN, LOW);
  poweron=true;
  Wire.begin();        // Join the I2C bus as a master
  startDac1();
  delay(300);
  startDac2();
  delay(200);
  #endif ALWAYSON

  }  // End setup()


// Begin 24LC256 EEPROM section

void writeData(int device, unsigned int add, byte data) // writes a byte of data 'data' to the chip at I2C address 'device', in memory location 'add'
{
  Wire.beginTransmission(device);
  Wire.write((int)(add >> 8));   // left-part of pointer address
  Wire.write((int)(add & 0xFF)); // and the right
  Wire.write(data);
  Wire.endTransmission();
  delay(10);
}

byte readData(int device, unsigned int add) // reads a byte of data from memory location 'add' in chip at I2C address 'device'
{
  byte result;  // returned value
  Wire.beginTransmission(device); //  these three lines set the pointer position in the EEPROM
  Wire.write((int)(add >> 8));   // left-part of pointer address
  Wire.write((int)(add & 0xFF)); // and the right
  Wire.endTransmission();
  Wire.requestFrom(device,1); // now get the byte of data...
  result = Wire.read();  
  return result; // and return it as a result of the function readData
}

// End 24LC256 section

// -------------------------------------loop section------------------------------------------

void loop() {

  // Print the sample rate, input and lock (once every "INTERVAL_SAMPLE" time)
  if((millis() - displayMillis > INTERVAL_SAMPLE*1000)&&!selectMode&&poweron) {
    displayMillis = millis(); // Saving last time we display sample rate

    Status=readStatus();  // Read status register
    if(Status&B00000100) SPDIFValid=true; // Determine if valid spdif data
    else SPDIFValid=false;                // SR calculation depends on I2S or SPDIf

    // Update the sample rate display if there is a lock (otherwise, SR=0)
    myGLCD.setFont(DotMatrix_M);
    myGLCD.setColor(0, 255, 0);

    if(Status&B00000001){        // There is a lock in a signal
      sr = sampleRate();
      if(Status&B00001000) {      // Determines if DSD
        sr*=64;                  // It is DSD, the sample rate is 64x
        myGLCD.print("DSD    ", 220, 215);
        if(SRExact==true){              
          myGLCD.print("         ", 220, 185);
          myGLCD.printNumI(sr, 220, 185);    // Print DSD sample rate in exact format
        }
        else                                 // Print DSD sample rate in nominal format
        if(sr>6143000)
          myGLCD.print("6.1 MHz     ", 220, 185);
        else
          if(sr>5644000)
            myGLCD.print("5.6 MHz     ", 220, 185);
          else
            if(sr>3071000)
              myGLCD.print("3.0 MHz     ", 220, 185);
            else
              if(sr>2822000)
                myGLCD.print("2.8 MHz     ", 220, 185);
              else
                myGLCD.print("Unknown     ", 220, 185);
      }
      else {                    // If not DSD then it is I2S or SPDIF
        if(SPDIFValid){           // If valid spdif data, then it is spdif
          myGLCD.print("S/PDIF ", 220, 215);
          if(SRExact==true){        // Print PCM sample rate in exact format
            myGLCD.printNumI(sr, 220, 185);
          }
          else                     // Print PCM sample rate in nominal format
          if(sr>191900)
            myGLCD.print("192 KHz     ", 220, 185);
          else
            if(sr>176300)
              myGLCD.print("176 KHz     ", 220, 185);
            else
              if(sr>95900)
                myGLCD.print("96 KHz      ", 220, 185);
              else
                if(sr>88100)
                  myGLCD.print("88 KHz      ", 220, 185);
                else
                  if(sr>47900)
                    myGLCD.print("48 KHz      ", 220, 185);
                  else
                    myGLCD.print("44.1 KHz    ", 220, 185);
        }

        else {
          myGLCD.print("PCM    ", 220, 215);  // Otherwise it is PCM
          if(SRExact==true){        // Print PCM sample rate in exact format
            myGLCD.printNumI(sr, 220, 185);
          }
          else                     // Print PCM sample rate in nominal format
          if(sr>383900)
            myGLCD.print("384 KHz     ", 220, 185);
          else
            if(sr>352700)
              myGLCD.print("352 KHz     ", 220, 185);
            else
              if(sr>191900)
                myGLCD.print("192 KHz     ", 220, 185);
              else
                if(sr>176300)
                  myGLCD.print("176 KHz     ", 220, 185);
                else
                  if(sr>95900)
                    myGLCD.print("96 KHz      ", 220, 185);
                  else
                    if(sr>88100)
                      myGLCD.print("88 KHz      ", 220, 185);
                    else
                      if(sr>47900)
                        myGLCD.print("48 KHz      ", 220, 185);
                          else
                            if(sr>44000)
                              myGLCD.print("44.1 KHz    ", 220, 185);
                            else
                              myGLCD.print("32 KHz      ", 220, 185);
        }
      }
    }
    else {                       // There is no lock we print input selection
      myGLCD.setFont(DotMatrix_M);
      myGLCD.setColor(255, 0, 0);
      myGLCD.print("No Lock     ", 220, 185);
      myGLCD.print("            ", 220, 215);
    }

     // Heatbeat to show that the software is still running...

    myGLCD.setFont(SmallFont);
    myGLCD.setColor(255, 0, 0);
    if(pulse++%2){
      myGLCD.print("d", 390, 0);
    }
    else {
      myGLCD.print("p", 390, 0);
    }
  }

  /*
  The following code is for the remote control. It can handle all the codes generated by a NEC 
   remote control except code "0" has been designated (by me) for "repeat" and code "255" designated
   (by me)as an error code.
   
   In addition, the code ignores the repeat code by default, except if designated for repeat. In
   this case the keys for up and down volume have been designated for repeat.
   
   You have to discover what codes your remote uses (use the piece of code below that displays the 
   detected IR code) and change the CASE statements accordingly.
   */

  while(digitalRead(REMOTEPIN)==LOW){
    if((IRkey=getIRkey())==255){
      // Do nothing
    }
    else { 
      if(IRkey==0) {            // Repeat code                           
        if(previousIRkey==6||previousIRkey==12) {
          IRkey=previousIRkey;  // Repeat code, only for specified keys. Add additional keys to 
          // or statement in is if condition
        }
        else {
          // Do nothing. No repeat for the rest of the keys
        }
      }
      else {                    // Not a repeat code, it is a new command
        previousIRkey=IRkey;    // Remember the key in case we want to use the repeat code
      }
    }

    // start temp code to determine IR codes - comment out for final version    
    myGLCD.setFont(Sinclair_S);
    myGLCD.setColor(255, 255, 255);
    if (IRkey != 255){
      myGLCD.print("   ", 350, 0);
      myGLCD.printNumI(IRkey, 350, 0);
    }
    // end temp code to determine IR codes - comment out for final version


    switch(IRkey){
      // case 0 and 255 are "valid" cases from the code, but we do nothing in this switch statement

    case 22:  // 22 is the eject button, used to toggle the power of the DAC
      if (poweron==false)                // Check if already powered on
      {
        digitalWrite(POWERPIN, HIGH);
        digitalWrite(DIMMPIN, LOW);
        poweron=true;
        Wire.begin();        // Join the I2C bus as a master
        startDac1();
        delay(300);
        startDac2();
        delay(200);
      }
      else {
        digitalWrite(POWERPIN, LOW);
        digitalWrite(DIMMPIN, HIGH);
        digitalWrite(SELECTI2SPIN, HIGH);
        poweron=false;
        myGLCD.clrScr();
        //insert code here to also power off the screen's backlight
      }
      break;

    case 12:  // 12 is the up key, we will use for volume up
      if (currAttnu>MINATTNU && poweron)                // Check if not already at minimum attenuation
      {
        if(dimmed) {
          rampUp();                          // Disengage dim
          dimmed=false;
        }

        currAttnu-=2;                      // Decrease attenuation 1 dB (increase volume 1 db)
        setSabreVolume(currAttnu);         // Write value into registers
        dispVol(currAttnu);
      }
      break;

    case 6:  // 6 is the down key, we will use for volume down
      if (currAttnu<MAXATTNU && poweron)              // Check if not already at maximum attenuation
      {
        if(dimmed) {
          rampUp();                        // Disengage dim
          dimmed=false;
        }

        currAttnu+=2;                      // Increase 1 dB attenuation (decrease volume 1 db)
        setSabreVolume(currAttnu);         // Write value into registers

        dispVol(currAttnu);

      }
      break;

    case 2:  // 2 is the center key. This is the soft mute -dim feature
      if(dimmed && poweron){                          // Back to previous volume level 1 db at a time
        rampUp();                          // Disengage dim
        dimmed=false;
      }
      else {
        if(DIM>=currAttnu && poweron) {               // only DIM if current attenuation is lower
          setSabreVolume(DIM);             // Dim volume

          dispVol(currAttnu);

          dimmed=true;
        }
      }
      break;

    case 65:  // 65 is the 1 key. Select source no. 1
      if(poweron) {
        input=0;
        input%=ICHO;
        setAndPrintInput(input);
      }
      break;

    case 77:  // 77 is the 2 key. Select source no. 2
      if(poweron) {
        input=1;
        input%=ICHO;
        setAndPrintInput(input);
      }
      break;

    case 69:  // 69 is the 3 key. Select source no. 3
      if(poweron) {
        input=2;
        input%=ICHO;
        setAndPrintInput(input);
      }
      break;
   
   #ifdef BUFFALO3
   case 66:  // 66 is the 4 key. Select source no. 4
      if(poweron) {
        input=3;
        input%=ICHO;
        setAndPrintInput(input);
      }
      break;

    case 78:  // 78 is the 5 key. Select source no. 5
      if(poweron) {
        input=4;
        input%=ICHO;
        setAndPrintInput(input);
      }
      break;

    case 70:  // 70 is the 6 key. Select source no. 6
      if(poweron) {
        input=5;
        input%=ICHO;
        setAndPrintInput(input);
      }
      break;

    case 67:  // 67 is the 7 key. Select source no. 7
      if(poweron) {
        input=6;
        input%=ICHO;
        setAndPrintInput(input);
      }
      break;

    case 79:  // 79 is the 8 key. Select source no. 8
      if(poweron) {
        input=7;
        input%=ICHO;
        setAndPrintInput(input);
      }
      break;

    case 71:  // 71 is the 9 key. Select source no. 9 (USB 1)
      if(poweron) {
        input=8;
        input%=ICHO;
        setAndPrintInput(input);
      }
      break;

    case 8:  // 8 is the 10+ key. Select source no. 10 (USB 2)
      if(poweron) {
        input=9;
        input%=ICHO;
        setAndPrintInput(input);
      }
      break;
    #endif BUFFALO3

    case 64:  // 64 is the ENTER key. Shift between source / settings / etc.
      if(poweron) {
      selectMode=true;          // Now we are in select mode
      removeSelectBar();        // Clear boxes etc.
      debounceMillis=millis();  // Start debounce timer
      selectMillis=millis();    // Start being-in-select-mode timer

      select++;  // Move to next select option
      myGLCD.setColor(255, 255, 0);

      switch(select%(MAXPARAM+3)){
      case VOL:
        break;
      case INP:
        myGLCD.drawRect(198, 23, 380, 90); // Input
        myGLCD.drawRect(199, 24, 379, 89); // Input
        break;
      case INF:
        myGLCD.drawRect(0, 0, 230, 19); // Input format
        myGLCD.drawRect(1, 1, 229, 18); // Input format
        break;
      case FIR:
        myGLCD.drawRect(0, 19, 162, 57); // FIR
        myGLCD.drawRect(1, 20, 161, 56); // FIR
        break;
      case IIR:
        myGLCD.drawRect(0, 57, 162, 76); // IIR
        myGLCD.drawRect(1, 58, 161, 75); // IIR
        break;
      case DPL:
        myGLCD.drawRect(0, 76, 162, 114); // DPLL
        myGLCD.drawRect(1, 77, 161, 113); // DPLL
        break;
      case QTZ:
        myGLCD.drawRect(0, 114, 162, 133); // Qtz
        myGLCD.drawRect(1, 115, 161, 132); // Qtz
        break;
      case NCH:
        myGLCD.drawRect(0, 133, 162, 152); // Notch
        myGLCD.drawRect(1, 134, 161, 151); // Notch
        break;
      case PLM:
        myGLCD.drawRect(0, 152, 162, 190); // DPLL Mode
        myGLCD.drawRect(1, 153, 161, 189); // DPLL Mode
        break;
      case OSF:
        myGLCD.drawRect(0, 190, 195, 209); // Oversampling
        myGLCD.drawRect(1, 191, 194, 208); // Oversampling
        break;
      case SRF:
        myGLCD.drawRect(0, 209, 195, 228); // SR
        myGLCD.drawRect(1, 210, 194, 227); // SR
        break;
      }  // End of switch
      }
      break;
      
      case 25:  // 25 is the left key.
      if(poweron) {
      irdir=1;    
      }
      break;
      
      case 17:  // 17 is the right key.
      if(poweron) {
      irdir=2;  
      }
      break;
    }
  } // End of remote control code

  /*
  To debounce the switch, we detect the first instance of the "click" and then ignore
   the switch: first during the switch "bouncing" time and then while the switch remains
   pressed because one cannot lift the finger so fast.
   We want to register a single "click" per push of the switch
   */

  if((digitalRead(SELECTPIN)==LOW)&&((millis()-debounceMillis)>INTERVAL_SWITCHBOUNCE)) {

    if (poweron==false)                // Check if already powered on
    {
      digitalWrite(POWERPIN, HIGH);
      digitalWrite(DIMMPIN, LOW);
      poweron=true;
      startDac1();
      startDac2();        
    }
    else {
      selectMode=true;          // Now we are in select mode
      removeSelectBar();         // Clear boxes etc.
      debounceMillis=millis();  // Start debounce timer
      selectMillis=millis();    // Start being-in-select-mode timer

      select++;  // Move to next select option
      myGLCD.setColor(255, 255, 0);

      switch(select%(MAXPARAM+3)){
      case VOL:
        break;
      case INP:
        myGLCD.drawRect(198, 23, 380, 90); // Input
        myGLCD.drawRect(199, 24, 379, 89); // Input
        break;
      case INF:
        myGLCD.drawRect(0, 0, 230, 19); // Input format
        myGLCD.drawRect(1, 1, 229, 18); // Input format
        break;
      case FIR:
        myGLCD.drawRect(0, 19, 162, 57); // FIR
        myGLCD.drawRect(1, 20, 161, 56); // FIR
        break;
      case IIR:
        myGLCD.drawRect(0, 57, 162, 76); // IIR
        myGLCD.drawRect(1, 58, 161, 75); // IIR
        break;
      case DPL:
        myGLCD.drawRect(0, 76, 162, 114); // DPLL
        myGLCD.drawRect(1, 77, 161, 113); // DPLL
        break;
      case QTZ:
        myGLCD.drawRect(0, 114, 162, 133); // Qtz
        myGLCD.drawRect(1, 115, 161, 132); // Qtz
        break;
      case NCH:
        myGLCD.drawRect(0, 133, 162, 152); // Notch
        myGLCD.drawRect(1, 134, 161, 151); // Notch
        break;
      case PLM:
        myGLCD.drawRect(0, 152, 162, 190); // DPLL Mode
        myGLCD.drawRect(1, 153, 161, 189); // DPLL Mode
        break;
      case OSF:
        myGLCD.drawRect(0, 190, 195, 209); // Oversampling
        myGLCD.drawRect(1, 191, 194, 208); // Oversampling
        break;
      case SRF:
        myGLCD.drawRect(0, 209, 195, 228); // SR
        myGLCD.drawRect(1, 210, 194, 227); // SR
        break;
      }  // End of switch
    }
  }

  // For the rotary encoder and the remote control. The control depends whether in display mode or select mode
  
  int dir=RotEncoder();
  
  if(dir==1 || dir==2 || irdir==1 || irdir==2)
  {
    delay(INTERVAL_BOUNCE);  // debounce by waiting INTERVAL_BOUNCE time

    switch(select%(MAXPARAM+3)){
    case VOL:  // Volume adjustment
      dimmed=false;                          // Disengage dim
      if (dir==1 || irdir==1)  // CCW
      {
        if (currAttnu<MAXATTNU && poweron)              // Check if not already at maximum attenuation
        {
          currAttnu+=2;                      // Increase 1 dB attenuation (decrease volume 1 db)
          setSabreVolume(currAttnu);         // Write value into registers
          dispVol(currAttnu);
          irdir=0;
        }
      }
      else                                   // If not CCW, then it is CW
      {
        if (currAttnu>MINATTNU && poweron)              // Check if not already at minimum attenuation
        {
          currAttnu-=2;                      // Decrease attenuation 1 dB (increase volume 1 db)
          setSabreVolume(currAttnu);         // Write value into registers
          dispVol(currAttnu);
          irdir=0;
        }
      }
      break;
    case INP:  // Input selection
      if (dir==1 || irdir==1) 
      {input--;  // CCW
      irdir=0;}
      else input++;
      input%=ICHO;
      setAndPrintInput(input);
      selectMillis=millis();  // Reset being-in-select-mode timer
      irdir=0;
      break;
    case INF:  // Input format selection
      if (dir==1 || irdir==1) {
      settings[input][FORMATVAL]--;  // CCW      
      irdir=0; }
      else settings[input][FORMATVAL]++;
      setAndPrintInputFormat(settings[input][FORMATVAL]%FORMATCHO);
      selectMillis=millis();  // Reset being-in-select-mode timer
      irdir=0;
      break;
    case FIR:  // FIR filter selection
      if (dir==1 || irdir==1) {
      settings[input][FIRVAL]--;  // CCW
      irdir=0; }
      else settings[input][FIRVAL]++;
      setAndPrintFirFilter(settings[input][FIRVAL]%FIRCHO); 
      selectMillis=millis();  // Update being-in-select-mode timer
      irdir=0;
      break;
    case IIR:  // IIR filter selection
      if (dir==1 || irdir==1) {
      settings[input][IIRVAL]--;  // CCW
      irdir=0;  }      
      else settings[input][IIRVAL]++;
      setAndPrintIirFilter(settings[input][IIRVAL]%IIRCHO);
      selectMillis=millis();  // Update being-in-select-mode timer
      irdir=0;
      break;
    case QTZ:  // Quantizer selection
      if (dir==1 || irdir==1) {
      settings[input][QUANVAL]--;  // CCW
      irdir=0; }
      else settings[input][QUANVAL]++;
      setAndPrintQuantizer(settings[input][QUANVAL]%QUANCHO);
      selectMillis=millis();  // Reset being-in-select-mode timer
      irdir=0;
      break;
    case DPL:  // DPLL setting: separate selection for SPDIF and I2S
      if (dir==1 || irdir==1) {      
      settings[input][DPLLVAL]--;  // CCW
      irdir=0;  }
      else settings[input][DPLLVAL]++;
      setAndPrintDPLL(settings[input][DPLLVAL]%DPLLCHO);
      selectMillis=millis();  // Reset being-in-select-mode timer
      irdir=0;
      break;
    case NCH:  // Notch selection
      if (dir==1 || irdir==1) {
      settings[input][NOTCHVAL]--;  // CCW
      irdir=0;  }
      else settings[input][NOTCHVAL]++;
      setAndPrintNotch(settings[input][NOTCHVAL]%NOTCHCHO);
      selectMillis=millis();  // Reset being-in-select-mode timer
      irdir=0;
      break;
    case PLM:  // DPLL mode selection
      if (dir==1 || irdir==1) {
      settings[input][PLMVAL]--;  // CCW
      irdir=0;  }
      else settings[input][PLMVAL]++;
      setAndPrintDPLLMode(settings[input][PLMVAL]%PLMCHO);
      selectMillis=millis();  // Reset being-in-select-mode timer
      irdir=0;
      break;
    case OSF:  //OSF Bypass -Since this is a toggle, we just call the function
      if (dir==1 || irdir==1) {
      settings[input][OSFVAL]--;  // CCW
      irdir=0;  }
      else settings[input][OSFVAL]++;
      setAndPrintBypassOSF(settings[input][OSFVAL]%OSFCHO);
      selectMillis=millis();  // Reset being-in-select-mode timer
      irdir=0;
      break;
    case SRF:  //Sampler Rate format -Since this is a toggle, we just call the function
      if (dir==1 || irdir==1) {
      settings[input][SRFVAL]--;  // CCW
      irdir=0;  }
      else settings[input][SRFVAL]++;
      setAndPrintSRFormat(settings[input][SRFVAL]%SRFCHO);
      selectMillis=millis();  // Reset being-in-select-mode timer
      irdir=0;
      break;

    }  // End of (rotary encoder) switch

  }  // End of if(rotating)

  // When the being-in-select mode timer expires, we revert to volume/display mode
  if(selectMode&&millis()-selectMillis > INTERVAL_SELECT*1000){
    selectMode=false;  // No longer in select mode
    removeSelectBar(); // "Removes" the select bar by printing blanks
    select=VOL;        // Back to volume mode
    writeSettings();   // Write new settings into EEPROM, including current input
  } 
} // End of loop()


// Begin new rotary encoder code (no longer using interrupts)

int state0[]={0,0,0,0};
int RotEncoder() {
  int encc = (digitalRead(VOLUPPIN) << 1) + digitalRead(VOLDOWNPIN);
  int out=4;

  int state1[]={1,0,2,3};
  int state2[]={0,2,3,1};
  int state3[]={2,3,1,0};
  int state4[]={3,1,0,2};
  int state5[]={0,1,3,2};
  int state6[]={1,3,2,0};
  int state7[]={3,2,0,1};
  int state8[]={2,0,1,3};

  if (state0[3] != encc) {
    for (int i=0;i<4;i++) {
      if (i<3) state0[i]=state0[i+1];
      else state0[i]=encc;
    }
    
    if ((state1[0] == state0[0] && state1[1] == state0[1] && state1[2] == state0[2] && state1[3] == state0[3])||
        (state2[0] == state0[0] && state2[1] == state0[1] && state2[2] == state0[2] && state2[3] == state0[3])||
        (state3[0] == state0[0] && state3[1] == state0[1] && state3[2] == state0[2] && state3[3] == state0[3])||
        (state4[0] == state0[0] && state4[1] == state0[1] && state4[2] == state0[2] && state4[3] == state0[3])) 
    out = 1;
    else if ((state5[0] == state0[0] && state5[1] == state0[1] && state5[2] == state0[2] && state5[3] == state0[3])||
             (state6[0] == state0[0] && state6[1] == state0[1] && state6[2] == state0[2] && state6[3] == state0[3])||
             (state7[0] == state0[0] && state7[1] == state0[1] && state7[2] == state0[2] && state7[3] == state0[3])||
             (state8[0] == state0[0] && state8[1] == state0[1] && state8[2] == state0[2] && state8[3] == state0[3])) 
    out = 2;
    else
    out = 3;
  }
  return out;
}
